#!/usr/bin/tclsh
#
# Velometrik Arbeitsstationsdeamon
#
# Ermöglicht die Steuerung einer Arbeitsstation mittels Browser.
#
# Author: Siegmar Müller
#
# Historie:
# ??.06.2019 Siegmar Müller Testversion fertiggestellt
# 05.05.2022 Siegmar Müller vmkstationd3 begonnen
# 01.11.2023 Siegmar Müller V 3.2 begonnen
# 30.12.2023 Siegmar Müller Globale Daten explizit mit :: versehen
# 13.01.2024 Siegmar Müller V 3.3 mit Recorder begonnen
# 24.01.2024 Siegmar Müller V 3.2.2 fertig
# 08.03.2024 Siegmar Müller V 3.3.0 (Recorder, App dekubitus) fertig
# 08.03.2024 Siegmar Müller V 3.4 begonnen
#

# Die Versionsnummer bezieht sich auf das gesamte Paket (nicht nur auf diese Datei).
# Gesondert versioniert sind die Apps und die htdocs.
set ::VERSION "3.4.0"

set ::dir [file dirname [info script]]
if {[string equal -length 1 $::dir "."]} {
    set ::dir "$env(PWD)[string range $::dir 1 end]"
}

# Globale Daten
set ::CONF_LOGFILE "$::dir/vmkstationd.log"; # Über Kommndozeile änderbar:
# Wird evtl. von Einträgen PORT bzw. LOGLEVEL in der Konfigurationsdatei überschrieben:
set ::CONF_PORT 80; # Kann zusätzlich über die Kommandozeile eingestellt werden
set ::CONF_LOGLEVEL notice
if {[string index $::VERSION end] in {A U}} {; # Alpha oder unfertige Version
    set ::CONF_LOGLEVEL debug
}
set ::PIEPSER 0; # Piepserplatine ist vorhanden (Wird nach erfolgreicher Initialisierung 1)
# Wird evtl. von gleichnamigen Einträgen in der Konfigurationsdatei überschrieben:
set ::BGCOLOR #404040
set ::COLORCONTRAST 0
set ::JPEGOPTIONS [list -quality 60 -res 12 -grid no -frame 0]
set ::WS_URL "https://ws-1-1.velometrik.de/vmkservice"
set ::WS_VERSION ""; # wird beim Webservice abgefragt
set ::STATION_NR 0; # Nicht registrierte Station. Das wird bei der Registierung in der Konfigdatei überschrieben.


# Start nur als root oder Angehöriger der Gruppe staff
set ::USER [exec id -n -u]
if {"$USER" != "root"} {
    set GROUPS [exec id -n -G]
    if {!("staff" in $GROUPS)} {
        puts stderr "vmkstationd must be executed by root or member of group staff"
        exit 1
    }
}

# Workaround für ein Problem, das mit tcllib1.19 aufgetreten ist:
if {![info exists ::env(USER)]} {
    set ::env(USER) root
}
if {![info exists ::env(user)]} {
    set ::env(user) $::env(USER)
}
#Ende: Workaround


# (weitere) globale Daten
# Bereits gesetzt: ::dir Stammverzeichnis von vmkstationd
set ::SODIR "." ; # Verzeichnis der shared objects (wird noch ermittelt)
set ::BINDIR "/usr/local/bin" ; # Verzeichnis der ausführbaren Dateien (wird noch ermittelt)
set ::PORTALSERVER "" ; # Wird vom Webservice geliefert
# (siehe auch setup/vmkstationd.conf)
# LAN beinhaltet WLAN
set ::STATUSLEVELS [list OFF POWER LAN INET WEBSERVICE LOGGED_IN]

#{{{ Konfiguration holen
set CONFIGFILE ""
if {[file exists /usr/local/etc/vmkstationd.conf]} {
    set CONFIGFILE "/usr/local/etc/vmkstationd.conf"
} elseif {[file exists $::dir/vmkstationd.conf]} {
    set CONFIGFILE "$::dir/vmkstationd.conf"
}
if {"$CONFIGFILE" == ""} {
    set CONFIGFILE "<defaults>"
} else {
    if {[catch {source $CONFIGFILE} msg]} {
        puts stderr "Error in Configfile: $msg"
        exit 1
    }
    if {[info exists PORT]} {
        set CONF_PORT $PORT
    }
    if {[info exists LOGLEVEL]} {
        set CONF_LOGLEVEL $LOGLEVEL
    }
}
#}}} Konfiguration holen

#{{{ Velobox Konfiguration
namespace eval ::VELOBOX_CONF {
    # Die erwarteten Variablen mit ihren Vorgaben
    variable VERSION "0.0"
    variable LANG "en"
    # vlbupdate
    variable DOWNLOADSERVER ""
    variable UPDATEDIR ""
    # List of additional hardware components
    # e.g.: HARDWARE=PIEPSER,JIT35
    variable HARDWARE ""
    # Usage of this velobox
    # e.g. USAGE=
    variable USAGE "bikefitting"
    # Implementation for USAGE
    # e.g. IMPLEMENTATION=velometrik
    variable IMPLEMENTATION ""

    # Velobox Konfiguration einlesen
    # @param filename   Dateiname der Konfiguration
    proc readVeloboxConf {filename} {
        catch {
            set fd [open $filename "r"]
            while {[gets $fd line] >= 0} {
                if {[regexp {^[[:space:]]*([A-Za-z0-9]+)[[:space:]]*=[[:space:]]*([^[:space:]#]*)} $line match name value]} {
                    set [namespace current]::$name "$value"
                }
            }
            close $fd
        }
    }; # proc readVeloboxConf 

    # Tatsächliche Werte, sofern vorhanden, einlesen
    if {[file exists /usr/local/etc/velobox.conf]} {
        readVeloboxConf /usr/local/etc/velobox.conf
    }
}; # namespace eval ::VELOBOX_CONF
#}}} Velobox Konfiguration

# Hilfe ausgeben und beenden
proc help {} {; #{{{
    puts ""
    puts "Velometrik Arbeitsstationsdeamon V $::VERSION"
    puts "Startparameter:"
    puts "    -p <port>     Start auf Port <port> (Vorgabe $::CONF_PORT)"
    puts "    -L <logfile>  Logfile (Vorgabe: $::CONF_LOGFILE)"
    puts "                  ('' => Ausgabe an stderr)"
    puts "    -l <loglevel> Loglevel (error warn notice info debug)"
    puts "                  (Vorgabe: $::CONF_LOGLEVEL)"
    puts "    -V        Version ausgeben und beenden"
    puts "    -h        Hilfe ausgeben und beenden"
    puts "    -?        Hilfe ausgeben und beenden"
    puts ""
    exit
    #}}}
}; # proc help


#{{{ Kommandozeilenparameter übernehmen
set status option
foreach arg $argv {
    switch -- $status {
        option {
            switch -- $arg {
                -p {
                    set status "-p"
                }
                -L {
                    set status "-L"
                }
                -l {
                    set status "-l"
                }
                -V {
                    puts $VERSION
                    exit
                }
                -h {
                    help
                    exit
                }
                -? {
                    help
                    exit
                }
                default {
                    puts stderr "Unbekannte Option: $arg"
                    exit 1
                }
            }; # switch $arg
        }
        "-p" {
            set CONF_PORT $arg
            set status option
        }
        "-L" {
            set CONF_LOGFILE $arg
            set status option
        }
        "-l" {
            if {[lsearch -exact  [list "error" "warn" "notice" "info" "debug"] $arg] < 0} {
                puts stderr "Unbekanntes Loglevel: '$arg'"
                exit 1
            }
            set CONF_LOGLEVEL $arg
            set status option
        }
    }; # switch $status
}
if {"$status" != "option"} {
    puts stderr "Wert für Option $status fehlt."
    exit 1
}
#}}} Kommandozeilenparameter sind übernommen


# Pakete laden
foreach {pkg_tcl pkg_debian} {http tcllib
                              json tcllib
                              Thread tcl-thread
                              tls tcl-tls
                              tdbc tcl8.6-tdbc
                              tdbc::sqlite3 tcl8.6-tdbc-sqlite3} {
    if {[catch {package require $pkg_tcl} errmsg]} {
        puts stderr $errmsg
        puts stderr "Is $pkg_debian installed?"
        exit 1
    }
}


# Logging neu starten
# Letztes Logfile, falls vorhanden sichern.
# @param reason Grund für den Neustart zum Protokollieren
proc logRestart {reason} {
    if {[file exists $::CONF_LOGFILE]} {
        set fd [open $::CONF_LOGFILE "a"]
        puts $fd "[clock format [clock seconds]] logRestart ($reason): ${::CONF_LOGFILE}~ wird erstellt"
        close $fd
        file copy -force $::CONF_LOGFILE ${::CONF_LOGFILE}~
        file delete $::CONF_LOGFILE
    }
    # Das nächste mal um Mitternacht
    set seconds [expr {[clock add [clock scan 23:59] 1 minutes] - [clock seconds]}]
    set fd [open $::CONF_LOGFILE "a"]
    puts $fd "Next logRestart after $seconds s"
    close $fd
    after [expr {$seconds * 1000}] {logRestart midnight}
}; # proc logRestart 

logRestart "deamon_start"


#{{{ Startkode holen und ausführen
# CONF_LOGFILE wird jetzt benötigt.
if {![file isdirectory [file dirname $::CONF_LOGFILE]]} {
    if {[catch {exec mkdir -p [file dirname $::CONF_LOGFILE]} errmsg]} {
        puts stderr "Logdir kann nicht angelegt werden: $errmsg"
        set ::CONF_LOGFILE ""
    }
}
if {[file isdirectory $::dir/start]} {
    set startmodules [lsort [glob $::dir/start/*.tcl]]
    foreach file $startmodules {
        if {[catch {source $file} msg]} {
            puts stderr "vmkstationd start fehlgeschlagen: $msg."
            exit 1
        }
    }
} else {
    puts stderr "Verzeichnis $::dir/start existiert nicht."
}
#}}} Startkode holen und ausführen

if {[llength [info procs srvLog]] == 0} {; # srvLog ist nicht verfügbar
    puts stderr "vmkstationd: Start fehlgeschlagen (proc srvLog existiert nicht)."
    exit 1
}
# Ab hier sind Logmeldungen möglich.
setLoglevel $CONF_LOGLEVEL; # produziert die 1. Logmeldung
srvLog {} Notice "(Start vmkstationd $::VERSION hat begonnen)"
# Gesammelte Meldungen jetzt ausgeben
srvLog {} Notice "Start modules loaded: $startmodules"
srvLog {} Notice "Configfile: $CONFIGFILE"


#{{{ TCL-Bibliotheken holen (binär und reines TCL)

set machine [exec uname -m]
if {[file isdirectory $::dir/bin/$machine]} {
    set ::BINDIR $::dir/bin/$machine
}
if {[file isdirectory $::dir/lib]} {
    if {[catch {
        if {[file isdirectory $::dir/lib/$machine]} {
            # shared Objects
            set ::SODIR $::dir/lib/$machine
            foreach file [glob -nocomplain $::SODIR/libtcl${tcl_version}*.so] {
                load $file
                srvLog {} Notice "$file loaded"
            }
        } else {
            srvLog {} Warn "$::dir/lib/${machine}/ doesn't exist."
        }
        #   reine TCL-Bibliotheken
        foreach file [glob -nocomplain $::dir/lib/*.tcl] {
            source $file
            srvLog {} Notice "$file loaded"
        }
    } msg]} {
        srvLog {} Error "Loading ${file}: $msg"
    }
} else {
   srvLog {} Warn "$dir/lib/ doesn't exist."
}

if {[namespace exists ::piepser]} {; # Piepser mit LEDs initialisieren
    # Das scheitert, wenn es nicht auf einem Raspi ausgeführt wird.
    # Es scheitert nicht, wenn es auf einem Raspi ohne Piepserplatine ausgeführt wird.
    if {[set ::PIEPSER [::piepser::gpioSetup]]} {
        srvLog gpioSetup Notice "Piepser initialisiert"
    } else {
        srvLog gpioSetup Warn "Der Piepser wurde nicht aktiviert: $msg"
    }
}

#}}} TCL-Bibliotheken holen


# Workaround zum Unterbinden der unnötig zeitverzögernden Hostabfrage bei nicht erreichbarem DNS-Server
proc ::websocket::fconfigure {args} {
    if {[lindex $args 1] in {-peername -sockname}} {
        upvar #0 clientcontext[lindex $args 0] context
        set ipaddr $context(ipaddr)
        set port $context(port)
        srvLog {Workaround} Debug "fconfigure $args => ${ipaddr}:$port"
        return [list $ipaddr $ipaddr $port]
    }
    return [::fconfigure {*}$args]
}; # proc ::websocket::fconfigure

# Fehlerbehandlung für Threads
proc threaderror {thread_id errorInfo} {
    srvLog [namespace current] Error "Thread id=$thread_id: $errorInfo"
}
thread::errorproc [namespace current]::threaderror 

##################################################################################################
proc tracecommand {command op} {
    srvLog trace Debug "$op: $command"
}
# trace add execution ::websocket::upgrade enterstep tracecommand
##################################################################################################

set ::WSServer::HTTPD(default) "index.tcls"


### FIFO für lokalen Clientzugriff
# Benutzt zum Reagieren auf Änderungen im Netzwerk
# Nur zu experimentellen Zwecken: Loglevel ändern, Deamon abwürgen
#{{{

# Eine über den FIFO eingegangene Meldung baerbeiten
proc handleFifoRequest {} {
    set request [gets $::fd_fifo]
    srvLog FIFO Info "Request eingegangen: $request"
    if {[string equal -length 6 $request "dhcpcd"]} {
        # dhcpcd start/stop
        after 1500 ::wscomm::inetcheck
    } elseif {[string equal -length 5 "$request" "loglevel"]} {
        setLoglevel [regsub {loglevel *} $request {}]
    } elseif {"$request" == "terminate"} {
        srvLog FIFO Notice "\"terminate\" empfangen."
        exit
    }
}; # proc handleFifoRequest 

if {[catch {
        exec /bin/mkdir -p -m 775 /var/local/run/vmkstationd
        if {![file exists /var/local/run/vmkstationd/vmkstationd.fifo]} {
            exec /usr/bin/mkfifo /var/local/run/vmkstationd/vmkstationd.fifo
        }
        set ::fd_fifo [open /var/local/run/vmkstationd/vmkstationd.fifo r+]
        fconfigure $::fd_fifo -blocking 0
        fileevent $::fd_fifo readable handleFifoRequest 
    } result]} {
    srvLog {} Error "FIFO-Kommunikation ist nicht möglich: $result"
} else {
    srvLog {} Notice "Lausche auf /var/local/run/vmkstationd/vmkstationd.fifo"
}

###}}} FIFO für lokalen Clientzugriff


### Der Kernel
namespace eval ::kernel {; #{{{

    srvLog {::kernel} Notice "::kernel wird initialisiert"

    ### Basis-Kernelmodule für alle Anwendungen aus /lib/kernel laden
    # (Die anwendungsspezifischen Module können sich auf das Vorhandensein dieser Module verlassen.)
    set mods1_loaded [list]; # Die geladenen Module müssen anschließend initialisiert werden.
    foreach tclfile [glob $::dir/lib/kernel/*.tcl] {
        set mod_loaded ""; # Das muß das Modul beim Laden setzen.
        
        if {[catch {source $tclfile} load_error]} {
            srvLog {::kernel} Error "$load_error\n${::errorInfo}"
            continue
        }
        if {"$mod_loaded" == ""} {
            srvLog {::kernel} Error "mod_loaded nicht gesetzt von $tclfile"
        } else {
            lappend mods1_loaded $mod_loaded
            srvLog {::kernel} Notice "Modul $mod_loaded geladen aus $tclfile"
        }
    }
    # Module initialisieren
    foreach mod_loaded $mods1_loaded {
        if {"[info procs ${mod_loaded}::init]" != ""} {
            if {[catch {${mod_loaded}::init} init_error]} {
                srvLog {::kernel} Error $init_error
            } else {
                srvLog {::kernel} Info "Modul $mod_loaded initialisiert"
            }
        } else {
            srvLog {::kernel} Warn "${mod_loaded}::init existiert nicht. Modul wird nicht initialisiert."
        }
    }

    ### Anwendungsspezifische Kernelmodule aus custom/kernel laden
    # (Die Anwendungen können sich auf das Vorhandensein dieser Module verlassen.)
    # (Kernel-)Module laden
    set mods2_loaded [list]; # Die geladenen Module müssen anschließend initialisiert werden.
    #TODO Duplikate in .../lib/kernel und .../custom/kernel vermeiden
    foreach tclfile [glob $::dir/custom/kernel/*.tcl] {
        set mod_loaded ""; # Das muß das Modul beim Laden setzen.
        
        if {[catch {source $tclfile} load_error]} {
            srvLog {::kernel} Error "$load_error\n${::errorInfo}"
            continue
        }
        if {"$mod_loaded" == ""} {
            srvLog {::kernel} Error "mod_loaded nicht gesetzt von $tclfile"
        } else {
            lappend mods2_loaded $mod_loaded
            srvLog {::kernel} Notice "Modul $mod_loaded geladen aus $tclfile"
        }
    }
    # Module initialisieren
    foreach mod_loaded $mods2_loaded {
        if {"[info procs ${mod_loaded}::init]" != ""} {
            if {[catch {${mod_loaded}::init} init_error]} {
                srvLog {::kernel} Error $init_error
            } else {
                srvLog {::kernel} Info "Modul $mod_loaded initialisiert"
            }
        } else {
            srvLog {::kernel} Warn "${mod_loaded}::init existiert nicht. Modul wird nicht initialisiert."
        }
    }

    srvLog {::kernel} Debug "Kernelmodule: [namespace children]"

    # Server und Kernelmodule starten
    proc start {} {; #{{{ (neu)
        srvLog {} Notice "Starte Arbeitsstationsdämon auf Port $::CONF_PORT ..."
        if {[catch {
            # HTTP/WS-Server starten
            ::WSServer::start "port $::CONF_PORT doc_root $::dir/htdocs"
            srvLog {} Notice "... gestartet."
            #TODO Alle Basis-Kernelmodule starten, sofern sie eine start-Prozedur haben.
            # Treiberüberwachung starten
            watchTTYACM::start
            srvLog {::kernel} Notice "::kernel gestartet"
        } errmsg]} {
            srvLog {::kernel} Error "::kernel nicht gestartet: $errmsg"
            puts "Kernel konnte nicht gestartet werden. (s. Logfile $::CONF_LOGFILE)"
            exit 1
        }
        #}}}
    }; # proc start 

    #{{{ Stationsstatus

    # Namen der Callbacks für Statusänderungen:
    variable status_handler_procnames [list]
    # Der aktuelle Statuslevel (Index in $::STATUSLEVELS)
    variable statuslevel 0
    
    # Callback für Statusänderung hinzufügen
    # Die Callback Prozedur wird mit dem neuen Status aufgerufen.
    # Der neue Status ist eine Liste mit Statusname und Statuslevel.
    # Nach dem Hinzufügen erfolgt sofort der erste Aufruf mit dem aktuellen Status.
    # @param cb_status_handler  Name des Callbacks
    proc addStatusHandler {cb_status_handler} {; #{{{
        variable status_handler_procnames 
        variable statuslevel
    
        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $status_handler_procnames $cb_status_handler] < 0} {
            lappend status_handler_procnames $cb_status_handler
            srvLog [namespace current]::addStatusHandler Debug "$cb_status_handler hinzugefügt"
        } else {
            srvLog [namespace current]::addStatusHandler Warn "$cb_status_handler bereits vorhanden"
        }
        $cb_status_handler [list [lindex $::STATUSLEVELS $statuslevel] $statuslevel]
        #}}}
    }; # proc addStatusHandler 
    
    # Callback für Statusänderung entfernen
    # @param cb_status_handler  Name des Callbacks
    proc removeStatusHandler {cb_status_handler} {; #{{{
        variable status_handler_procnames 
    
        set i [lsearch $status_handler_procnames $cb_status_handler]
        if {$i >= 0} {
            set status_handler_procnames [lreplace $status_handler_procnames $i $i]
            srvLog [namespace current]::removeStatusHandler Debug "$cb_status_handler entfernt"
        } else {
            srvLog [namespace current]::removeStatusHandler Warn "$cb_status_handler nicht gefunden"
        }
        #}}}
    }; # proc removeStatusHandler 
    
    # Status setzen/zurücksetzen
    # Schaltet den angegebenen Status ein oder aus
    # und bringt anschließend den höchsten Statuslevel zur Anzeige,
    # für den alle Vorgängerlevels eingeschaltet sind.
    # @param statusname s. ::STATUSLEVELS
    # @param on_off     0 oder 1
    proc setStationsstatus {statusname {on_off 1}} {; #{{{
        variable status_handler_procnames 
        variable statuslevel
    
        set i_status [lsearch $::STATUSLEVELS $statusname]
        if {$i_status < 0} {; # interner Fehler
            srvLog [namespace current]::setStationsstatus Warn "Unknown status '$statusname' ignored."
            return; # wird ignoriert.
        }
        if {!$on_off} {; # off
            # Der neue Status ist der vor dem ausgeschalteten.
            incr i_status -1
        }
        set statuslevel $i_status
        set status_neu [lindex $::STATUSLEVELS $i_status]
        signals::put STATUS $status_neu; 
        # Statusänderung über registrierte Callbacks verteilen
        foreach {status_handler} $status_handler_procnames {
            srvLog [namespace current]:setStationsstatus Debug "$status_handler {$status_neu $i_status}"
            $status_handler [list $status_neu $i_status]
        }
        srvLog [namespace current]::setStationsstatus Notice "Stationsstatus: {$status_neu $i_status}"
        #}}}
    }; # proc setStationsstatus 

    #}}} Stationsstatus in ::kernel verschieben

    #}}}
}; # ::kernel


::kernel::start
::kernel::setStationsstatus POWER


# Kommunikation mit dem Webservice
# Prüft die Internetverbindung und meldet sich beim Webservice
namespace eval ::wscomm {; #{{{
    variable local_ip ""
    variable netzwerk_aktiv 0
    variable internet_verfuegbar 0
    variable webservice_verfuegbar 0
    variable after_id ""; # für die Wiederholungsversuche

    # Lokales Netzwerkinterface testen.
    # varmsg    Name der Variablen zum Hinterlegen der Meldung
    # => 1  Das Interface koennte funktionieren.
    #    0  Es wurde ein Fehler festgestellt.
    proc localIfcheck {msgvar} {; #{{{
        upvar $msgvar msg
        variable local_ip
    
        set driver ""
        if {[catch {
                foreach line [split [exec ip -4 address] "\n"] {
                    if {[regsub {^ *inet ([0-9.]*)/.*$} $line {\1} temp]} {; # temp hat local_ip
                        set local_ip $temp
                        if {"$driver" == "lo"} {
                            continue
                        } elseif {[string match "eth*" $driver]} {; # ... hat Vorrang
                            break
                        }
                    }
                    if {[regsub {^.* ([a-z][a-z0-9]*): .*$} $line {\1} temp]} {; # temp hat driver
                        set driver $temp
                        continue
                    }
                }
            } errmsg]} {
                set msg "LOCAL_IP kann nicht ermittelt werden: $errmsg"
                return 0
        } else {
            if {"$driver" == "lo"} {
                set msg "Keine Netzwerkverbindung"
                ::kernel::setStationsstatus LAN 0
                return 0
            } else {
                set msg "$driver: LOCAL_IP=$local_ip"
                ::kernel::setStationsstatus LAN
                return 1
            }
        }
        #}}}
    }; # proc localIfcheck


    # HTTP-Request ausfuehren
    # httpUrl   Vollstaendiger Request
    # datavar   Variable zum Hinterlegen der Antwort bzw. der Fehlermeldung
    # =>    1 Request wurde ausgefuehrt, datavar enthaelt das Ergebnis
    #       0 datavar enthaelt eine Fehlermeldung oder ist leer, wenn die Fehlermeldung zuvor schon uebergeben wurde
    proc execHttpRequest {httpUrl datavar} {; #{{{
        upvar $datavar data
        variable last_ws_error
        variable last_user_error

        set ok 0
        if {[catch {
            set tokenHttp [::http::geturl $httpUrl]; # Wirft evtl. eine Exception
            # Verbindung konnte hergestellt werden, falls hier angekommen.
            ::http::wait $tokenHttp
            set err_msg ""
            # http-Antwort auswerten
            switch [::http::status $tokenHttp] {
                ok {
                    if {[::http::ncode $tokenHttp] == 200} {
                        set data [::http::data $tokenHttp]
                        set ok 1
                    } else {
                        set err_msg "#WS: [::http::code $tokenHttp]"
                    }
                }
                eof {
                    set err_msg "#EOF: Connection closed by Webservice."
                }
                error {
                    set err_msg "#ERROR: [::http::error $tokenHttp]"
                }
            }
            ::http::cleanup $tokenHttp

            if {!$ok} {
                # Exception erst hier werfen, damit ::http::cleanup noch ausgefuehrt wird.
                error $err_msg
            }
        } catchResult]} {
            srvLog {} Error $catchResult
        }
        return $ok
        #}}}
    }; # execHttpRequest


    # Abarbeitung eines Webservice Requests
    # request   Bezeichnung des Requests
    # lquery    Parameter als Liste [list parameter wert ...]
    # datavar   Variable fuer das Ergebnis
    # =>    1 Request wurde ausgefuehrt, datavar enthaelt das Ergebnis
    #       0 datavar enthaelt eine Fehlermeldung oder ist leer, wenn die Fehlermeldung zuvor schon uebergeben wurde
    proc execWsRequest {request lquery datavar} {; #{{{
        upvar $datavar data

        set parameter "[::http::formatQuery {*}$lquery]"
        if {[catch {
            srvLog {} Debug "Encrypting $parameter"
            set encrypted [vmkcrypt encrypt $parameter]
        } errmsg ]} {
            set data ""
            srvLog {} Error $errmsg
            return 0  
        }
        # Da keine Sonderzeichen mehr vorkommen koennen, kann das direkt zusammengebaut werden.
        #TODO Request kann nicht ausgeführt werden,
        # wenn STATION_NR == 0, d.h. nicht initialisiert ist.
        set parameter "sn=$::STATION_NR&req=$encrypted"
        srvLog {} Debug "execWsRequest: ${::WS_URL}/${request}?$parameter"
        return [execHttpRequest "${::WS_URL}/${request}?$parameter" data]
        #}}}
    }; # proc execWsRequest 


    # Ruft localIfcheck auf und prüft bei Erfolg den Internetzugang durch ping auf Google.
    # Wenn das erfolgreich war, wird dem Webserver die ermittelte local_ip mitgeteilt
    # und die zurückgegebene URL in ::PORTALSERVER gespeichert.
    # Falls diese Kette nicht fehlerfrei durchlaufen wurde, enthält ::PORTALSERVER einen Leerstring.
    # @param retry Anzahl der Wiederholungsversuche nach (10s) wenn Schnittstelle aktiv, aber keine Internetverbindung
    proc inetcheck {{retry 5}} {; #{{{
        variable local_ip
        variable netzwerk_aktiv
        variable internet_verfuegbar
        variable webservice_verfuegbar
        variable after_id

        set ::PORTALSERVER ""
        if {$after_id != ""} {
            after cancel $after_id
        set after_id ""
        }
        if {[localIfcheck nwmsg]} {
            # Lokale Schnittstelle ist aktiv.
            set netzwerk_aktiv 1
            srvLog {inetcheck} Notice "Lokale Netzwerkschnittstelle ist aktiv ($nwmsg)"
            # => Verfuegbarkeit des Internets pruefen
            #    Das geschieht mittels ping auf den Google DNS-Server 8.8.8.8.
            if {![catch {exec ping -4 -c1 8.8.8.8} result]} {
                srvLog {inetcheck} Notice "Internetzugang O.K."
                set internet_verfuegbar 1
                ::kernel::setStationsstatus INET
                # Antwortet der Webservice ?
                if {[execWsRequest ping {} wsAntwort]} {; # Er antwortet
                    srvLog {inetcheck} Debug "Webservice antwortet auf /ping mit \"$wsAntwort\"."
                    # Webservice Version festhalten
                    set dictWsAntwort [::json::json2dict $wsAntwort]
                    if {[dict exists $dictWsAntwort version]} {
                        set ::WS_VERSION [dict get $dictWsAntwort version]
                        srvLog {inetcheck} Notice "Webservice ${::WS_URL} hat die Version ${::WS_VERSION}."
                    }
                    set webservice_verfuegbar 1
                    ::kernel::setStationsstatus WEBSERVICE
                    # (Bei der Gelegenheit setzen wir gleich die IPadresse.)
                    if {[execWsRequest set [list station_nr $::STATION_NR what local_ip value $local_ip] wsAntwort]} {
                        srvLog {inetcheck} Debug "Webservice antwortet auf /set local_ip mit \"$wsAntwort\"."
                        if {[regexp {^{.*}$} $wsAntwort]} {; # JSON-Antwort
                            set dictWsAntwort [::json::json2dict $wsAntwort]
                            if {[dict exists $dictWsAntwort portalserver]} {
                                set ::PORTALSERVER [dict get $dictWsAntwort portalserver]
                                srvLog {inetcheck} Notice "Portalserver ist $::PORTALSERVER."
                                ::kernel::setStationsstatus LOGGED_IN
                            } else {
                                srvLog {inetcheck} Warn "Webservice meldet keinen Portalserver."
                                ::kernel::setStationsstatus LOGGED_IN 0
                            }
                        }
                    }
                    # Ab Webservice Version 1.6 nimmt dieser die Features und die Konfiguration der Velobox entgegen.
                    if {[regsub {^([0-9]+\.[0-9]+).*$} $::WS_VERSION {\1} ws_version] == 1} {
                        if {$ws_version >= 1.6} {
                            srvLog {inetcheck} Debug "Webservice Version >= 1.6"
                            # => features senden
                            set hardware [list]
                            foreach hwkomp [split $::VELOBOX_CONF::HARDWARE ","] {
                                lappend hardware \"[string map {\" \\\"} $hwkomp]\"
                            }
                            set apps [list]
                            foreach appcode [dict keys $::apps::apps_loaded] {
                                lappend apps $appcode
                                if [info exists ::apps::${appcode}::VERSION] {
                                    lappend apps [set ::apps::${appcode}::VERSION]
                                } else {
                                    lappend apps "0.0.0"
                                }
                            }
                            set json_apps [list]
                            foreach {app version} $apps {
                                lappend json_apps "{\"${app}\": \"$version\"}"
                            }
                            set features "\{\
                                \"VERSION\": \"$::VELOBOX_CONF::VERSION\",\
                                \"HARDWARE\": \[[join $hardware {, }]\],\
                                \"apps\": \[[join $json_apps {, }]\]\
                                \}"
                            if {[execWsRequest set [list station_nr $::STATION_NR what features value "$features"] wsAntwort]} {
                                srvLog {inetcheck} Debug "Webservice antwortet auf /set features mit \"$wsAntwort\"."
                                #TODO evtl. Fehler => entsprechendes srvLog
                            }
                            set htdocs "{}"
                            if {[file exists ${::dir}/htdocs/htdocs.json]} {
                                set fd [open ${::dir}/htdocs/htdocs.json r]
                                set json_htdocs [read $fd]
                                close $fd
                                if {[catch {::json::json2dict $json_htdocs}]} {
                                    srvLog {inetcheck} Error $result
                                } else {
                                    set htdocs [string map {"\n" ""} $json_htdocs]
                                }
                            }
                            set konfiguration "\{\
                                \"LANG\": \"$::VELOBOX_CONF::LANG\",\
                                \"DOWNLOADSERVER\": \"$::VELOBOX_CONF::DOWNLOADSERVER\",\
                                \"UPDATEDIR\": \"$::VELOBOX_CONF::UPDATEDIR\",\
                                \"USAGE\": \"$::VELOBOX_CONF::USAGE\",\
                                \"IMPLEMENTATION\": \"$::VELOBOX_CONF::IMPLEMENTATION\",\
                                \"htdocs\": $htdocs\
                                \}"
                            if {[execWsRequest set [list station_nr $::STATION_NR what konfiguration value "$konfiguration"] wsAntwort]} {
                                srvLog {inetcheck} Debug "Webservice antwortet auf /set konfiguration mit \"$wsAntwort\"."
                                #TODO evtl. Fehler => entsprechendes srvLog
                            }
                        }
                    }
                } else {; # Webservice antwortet nicht
                    srvLog {inetcheck} Warn "Webservice nicht verfügbar"
                    set webservice_verfuegbar 0
                    ::kernel::setStationsstatus WEBSERVICE 0
                }
            } else {; # Keine Antwort auf ping 8.8.8.8
                srvLog {inetcheck} Warn "Kein Internetzugang ($result)"
                set internet_verfuegbar 0
                ::kernel::setStationsstatus INET 0
            }
        } else {; # Fehler
            srvLog {inetcheck} Error $nwmsg
        set netzwerk_aktiv 0
        }
        if {$netzwerk_aktiv && !$internet_verfuegbar && $retry} {
            # Weiter probieren
            set after_id [after 30000 ::wscomm::inetcheck [expr $retry - 1]]
        } else {
            srvLog {inetcheck} Notice "Keine weiteren Verbindungsversuche zum Internet"
        }
        #}}}
    }; # proc inetcheck

    # Verfuegbarkeit des Netzwerkes und des Webservice pruefen
    # Vorsichtshalber etwas warten, falls der Bootvorgang noch nicht so weit ist.
    after 5000 ::wscomm::inetcheck

    #}}}
}; # namespace eval ::wscomm


### Die Anwendungen (Apps)
#
# Eine App befindet sich als namespace eval <appname> {...} in einer tcl-Datei in .../custom/apps/
# Als letzter Befehl (nach namespace eval{} wird 'set app_loaded <appname>' erwartet.)
# Nachdem alle Apps geladen sind, werden sie durch Aufruf der Prozedur <appname>::init initialisiert.
# Deren Fehlen wird als Fehler angesehen.
# Zu jeder App gehört weiterhin eine gleichnamige Websocketdomäne .../custom/wsserver/wsdomains/apps/<appname>.tcl.
# Sobald ein Client eine Websocketverbindung /apps/<appname> herstellt, wird das als Start interpretiert
# und führt zum Aufruf von ::apps::<appname>::start.
# Das Beenden der Websocketverbindung führt zum Aufruf von ::apps::<appname>::stop.
namespace eval ::apps {; #{{{
    variable apps_loaded [dict create]; # Schlüssel App, Wert Filename

    srvLog "[namespace current]" Notice "::apps werden initialisiert"

    # Fehlermeldung als wsevent in JSON erstellen
    # Die erstellten Schlüssel sind:
    #   "wsevent" mit Wert "error"
    #   "class" Fehlerklasse
    #   "source"  Fehlerquelle
    #   "nr"    Fehlernummer
    #   "msg"   Eine Fehlermeldung
    # Derzeit definierte Fehlernummern 
    #   1 Insufficient data (for analysis)
    #   2 Still recording
    #   3 Unstored recording(s)
    #   4 No session
    #   5 No finished recording
    # @param class_nr   Liste aus Fehlerklasse (internal|client|user) und nr
    # @param msg        Text
    proc createJSONError {class_nr source msg} {; #{{{
        srvLog [namespace current] Debug "createJSONError '$class_nr' '$msg'"
        set class [lindex $class_nr 0]
        set nr [lindex $class_nr 1]
        # Gültige Fehlerklasse?
        if {"$class" ni {internal client user}} {
            srvLog [namespace current] Debug "createJSONError Unknown error class: \"$class\""
        }
        set msg [string map {\" \\\"} $msg]
        set kvlist [list "wsevent" "error" "class" $class "source" "$source" "msg" $msg]
        if {"$nr" != ""} {
            lappend kvlist nr $nr
        }
        set json_error [::kvlist2json $kvlist]
        return $json_error
        #}}}
    }; # proc createJSONError 


    # App neu laden
    # @return Fertigmeldung
    proc reload {app} {; #{{{
        variable apps_loaded
 
        if {[catch {
            # Applikation stoppen, falls gerade am Laufen
            if {[info exists ::apps::${app}::started]} {
                set started [set ::apps::${app}::started]
            } else {
                set started 0
            }
            if {$started} {
                if {"[info procs ::apps::${app}::stop]" != ""} {
                    ::apps::${app}::stop
                    append msg "$app stopped\n"
                }
            }
            set tclfile [dict get $apps_loaded $app]
            set app_loaded ""
            source $tclfile
            if {"$app_loaded" == ""} {
                append msg "Error: app_loaded not set by $tclfile"
            } else {
                append msg "$app_loaded loaded from $tclfile"
                # Initialisieren
                if {"[info procs ${app_loaded}::init]" != ""} {
                    ${app_loaded}::init
                    append msg "\n$app_loaded initialisiert"
                } else {
                    append msg "\n${app_loaded}::init existiert nicht. App wird nicht initialisiert."
                }
                # neu starten, falls vorher auch gestartet
                if {$started} {
                    if {"[info procs ::apps::${app}::start]" != ""} {
                        ::apps::${app}::start
                        append msg "\n$app (re)started"
                    }
                }
            }
        } result]} {
            set msg "Error: $result"
        }
        return $msg
        #}}}
    }; # proc reload 


    ## Apps aus custom/apps laden und initialisieren
    # (Anwendungs-)Module laden
    foreach tclfile [glob $::dir/custom/apps/*.tcl] {
        set app_loaded ""; # Das muß das Modul beim Laden setzen.
        if {[catch {source $tclfile} load_error]} {
            srvLog [namespace current] Error "$load_error\n$errorInfo"
            continue
        }
        if {"$app_loaded" == ""} {
            srvLog [namespace current] Error "app_loaded nicht gesetzt von $tclfile"
        } else {
            dict set apps_loaded $app_loaded $tclfile
            srvLog [namespace current] Notice "$app_loaded geladen aus $tclfile"
        }
    }
    # Apps initialisieren
    foreach app_loaded [dict keys $apps_loaded] {
        if {"[info procs ${app_loaded}::init]" != ""} {
            if {[catch {${app_loaded}::init} init_error]} {
                srvLog [namespace current] Error $init_error
            } else {
                srvLog [namespace current] Info "$app_loaded initialisiert"
            }
        } else {
            srvLog [namespace current] Warn "${app_loaded}::init existiert nicht. App wird nicht initialisiert."
        }
    }

    #}}}
}; # ::apps


# Deamon erfolgreich gestartet
# Falls es einen Link auf eine Fallback-Version gibt, wird der nicht mehr gebraucht.
if {[file exists /usr/local/vmkstationd_last]} {
    file delete /usr/local/vmkstationd_last
    srvLog {} Notice "/usr/local/vmkstationd_last gelöscht."
}


# Behandlung von Hintergrundfehlern während vwait
# Wird aufgerufen, wenn der Interpreter in einen vorher nicht aufgefangenen Fehler läuft:
# @param msg    Meldung vom TCL-Kern
proc bgerror {msg} {
    global errorInfo

    srvLog bgerror Error "$msg\n$errorInfo"
}; # proc bgerror

vwait forever
exec stty echo

