#!/bin/bash
#
# create_vmkstationd_tar_gz.sh
#
# Erzeugt ein Installationspaket für den vmkstationd
# => create_vmkstationd_vlbupdate_tar_gz.sh erzeugt daraus ein Setup/Update Paket für vlbupdate.
#
# Dieses Skript befindet sich im Unterverzeichnis /devel
# und muß sich während der Ausführung ebenfalls dort befinden.
# Arbeitsverzeichnis ist das "vmkstiond3" enthaltende Verzeichnis.
# Der Aufruf erfolgt ohne Kommandozeilenparameter.
# Der Start erfolgt nach einer Rückfrage. Bei Abbruch wird mit Status 1 beendet.
# Die Ausführung muß durch root erfolgen, damit die Dateien den Besitzer und die Gruppe "root" erhalten können.
# Dazu wird alles temporär in ein Verzeichnis vmkstationd_V<VERSION> kopiert.
# Das Verzeichnis devel/ sowie Backups und Versionierungsdateien werden aus dem Archiv ausgeschlossen.
# Der Verzeichnisname ist gleichzeitig der Basisname des Tarfiles.
# Versionen mit Vorläufigkeitsanhang erhalten das aktuelle Datum.
# Ältere Installationspakete werden nach Rückfrage gelöscht.
#
# Velometrik GmbH
#
# Author: Siegmar Müller
# Historie:
#   28.10.2019  Siegmar Müller erstellt für vmkstationd
#   09.02.2023  Siegmar Müller umgebaut für vmkstationd3
#   25.10.2023  Siegmar Müller Kopfkommentar überarbeitet
#   18.01.2024  Siegmar Müller Stammverzeichnist ist vmkstationd[_V<version>]
#


# Nicht mehr benötigte temporäre Dateien/Verzeichnisse am Ende löschen
CLEANUP=true

# Ausführung als root, damit die Dateien Besitzer und Gruppe "root" bekommen können.
if [ $UID -ne 0 ]; then
    echo "Das Skript muß von root ausgeführt werden."
    su -c $0
    exit $?
fi

echo "Installationspaket ../vmkstationd/setup/vmkstationd_V<version>.tar.gz erstellen ..."

# Beendet das Skript mit dem angegeben Status
# und stellt zuvor nötigenfalls vmkstationd.tcl wieder her.
exitScript () {
    if [ -f vmkstationd/vmkstationd.tcl~ ]; then
        cp -fv vmkstationd/vmkstationd.tcl~ vmkstationd/vmkstationd.tcl
    fi
    [ $# -eq 0 ] && exit 1
     rm -fv vmkstationd/vmkstationd.tcl~
    exit $1
}


SCRIPT_DIR=$( cd -- "$( dirname $0 )" > /dev/null && pwd)
echo "SCRIPT_DIR = $SCRIPT_DIR"
if ! `echo ${SCRIPT_DIR} | grep -Eq 'vmkstationd(_V[0-9.]+)?/devel$'`; then
    echo "$( basename $0 ) muß sich im Verzeichnis ../vmkstationd[_V<version>]/devel befinden."
    exit 1
fi
VMKSD_DIR=`echo $SCRIPT_DIR | sed 's/\/devel$//'`
echo "VMKSD_DIR = $VMKSD_DIR"
cd $SCRIPT_DIR
cd .. # vmkstationd
cd ..
echo "Skriptausführung in `pwd`."
# Wir sind nun im Ausführungsverzeichnis.

# Die Versionsnummer wird dem Hauptskript (ohne Datumsanhang) entnommen.
VERSION=`grep '^set *::VERSION' ${VMKSD_DIR}/vmkstationd.tcl | sed -E 's/^.*([0-9]+\.[0-9]+\.[0-9]+_?[XAU]?).*$/\1/'`
[ ${PIPESTATUS[0]} -ne 0 ] && exit 1
echo "Gefundene Version: $VERSION"
# Gültige Version ?
echo $VERSION | grep '^[0-9][0-9]*\.[0-9][0-9]*\.[0-9][0-9]*[^ ]*$' > /dev/null
if [ $? -ne 0 ]; then
    echo "Die Version kann nicht ermittelt werden."
    echo "(Die Syntax scheint nicht zu stimmen.)"
    exit 1
fi
if `echo $VERSION | grep -q '_[XAU]$'`; then # Vorläufige Version
    # => aktuelles Datum einbauen.
    VERSION=${VERSION}`date +%Y%m%d`
    # Diese Version in das Skript hineinoperieren
    cp -fv ${VMKSD_DIR}/vmkstationd.tcl ${VMKSD_DIR}/vmkstationd.tcl~
    sed -E '/^ *set *VERSION/s/VERSION .*$/VERSION '"$VERSION"'; # modified by create_vmkstationd_tar_gz.sh/' ${VMKSD_DIR}/vmkstationd.tcl~ > ${VMKSD_DIR}/vmkstationd.tcl
    echo "Aktuelles Datum in unfertige Version eingebaut: $VERSION"
else
    echo "Fertige Version: $VERSION"
fi

# Alles muß in ein Verzeichnis mit eingeschlossener Versionsnummer gepackt werden.
# Das kann sich mit $VMKS_DIR beißen! => Eigenes tmp/ anlegen und dort weitermachen.
mkdir -p tmp
cd tmp
TMPDIR=vmkstationd_V${VERSION} # nicht vmkstationd_...
echo "Temp. Verzeichnis für vmkstationd: `pwd`/$TMPDIR"

echo -n "Create ${TMPDIR}.tar from vmkstationd (J|n): "
read ANSWER
[ "${ANSWER}" == "" ] && ANSWER="J"
[ "${ANSWER}" != "J" ] && { echo "abgebrochen"; exitScript 1; }

# Unwahrscheinlich aber nicht unmöglich:
if [ -d $TMPDIR ]; then
    read -p "Altes Verzeichnis `pwd`/$TMPDIR löschen?[Jn]: " A
    if [ "$A" == "n" ]; then
        echo "abgebrochen"
        exitScript 1
    fi
    rm -rf ./$TMPDIR
fi

# tarfile erstellen
cp ${VMKSD_DIR} -r $TMPDIR
chown -R root $TMPDIR
chgrp -R root $TMPDIR
echo
tar --exclude=${TMPDIR}/devel --exclude-backups --exclude-vcs -cvf ${TMPDIR}.tar ${TMPDIR}
echo
gzip -f ${TMPDIR}.tar
chown root ${TMPDIR}.tar.gz
chgrp root ${TMPDIR}.tar.gz
echo "${TMPDIR}.tar.gz erzeugt"

# tarfile nach ${VMKSD_DIR}/setup kopieren
# Vorhandene tars vorher löschen
TARS=`ls ${VMKSD_DIR}/setup/vmkstationd*.tar.gz 2>/dev/null`
if [ ${#TARS} -gt 0 ]; then
    echo -n "Alte tarfile(s) in ${VMKSD_DIR}/setup/ löschen? (J|n): "
    read ANSWER
    [ "${ANSWER}" == "" ] && ANSWER="J"
    [ "${ANSWER}" != "J" ] && { echo "abgebrochen"; exitScript 1; }
    for TAR in $TARS; do
        rm -v $TAR
    done
fi
mv -v ${TMPDIR}.tar.gz ${VMKSD_DIR}/setup
chmod a+w ${VMKSD_DIR}/setup/${TMPDIR}.tar.gz

# vmkstationd.tcl mit ursprünglicher Version wiederherstellen
if [ -f ${VMKSD_DIR}/vmkstationd.tcl~ ]; then
     cp -fv ${VMKSD_DIR}/vmkstationd.tcl~ ${VMKSD_DIR}/vmkstationd.tcl
    [ $? -ne 0 ] && exit 1
     rm -fv ${VMKSD_DIR}/vmkstationd.tcl~
fi
if `$CLEANUP`; then
    echo "Lösche temporäres Verzeichnis tmp/$TMPDIR"
    rm -rf ./$TMPDIR
    cd .. # 'raus aus tmp/
    if [ `ls tmp | wc -w` -eq 0 ]; then # tmp ist leer
        echo "tmp/ ist leer."
        rmdir tmp && echo "=> tmp/ entfernt."
    else
        echo "tmp/ ist nicht leer."
    fi
fi
echo "Installationspaket ${VMKSD_DIR}/setup/${TMPDIR}.tar.gz erzeugt"

