document.getElementById("ws_url").value = `ws://${window.location.hostname}:${window.location.port}/apps/sitzknochenanalyse`

/**
 * The function updates the image source of an element with the id "hockerbild" to prevent caching.
 */
function hockerbildAktualisieren () {
    // Der Zeitstempel verhindert die Benutzung des letzten Bildes im Cache.
    if (document.getElementById("JPEG_details").open)
        document.getElementById("hockerbild").src = "/hockerbild?rnd="+(new Date).getTime();
    if (document.getElementById("canvas_details").open)
        pimg.loadJPEG ();
}

/**
 * The function `handleWSMessage` is used to handle different types of WebSocket messages and update
 * the HTML content accordingly.
 * @param wsmessage - The `wsmessage` parameter is a string that represents a message received from a
 * WebSocket connection.
 * @returns In the code provided, there is no explicit return statement. Therefore, the function will
 * implicitly return undefined.
 */
function handleWSMessage (wsmessage) {
    var msg;

    try {
         msg = JSON.parse (wsmessage);
    } catch (error) {
        console.log (`Kein gültiger JSON-String: "${wsmessage}"`);
        return;
    }
    if (msg.wsevent == "hockerbild") {// Pressure image available as JPEG
	        hockerbildAktualisieren();
    } else if (msg.wsevent == "hockernorm") {//{{{ Pressure data as normed values
        // {"wsevent": "hockernorm", "imagetype": "ska_live", "n_rows": 16, "n_cols": 28, "values": [...]}
        var i;
        var v;

        for (i=0; i<msg.n_rows*msg.n_cols; ++i) {
            v = msg.values[i];
            let el = document.getElementById(`nv${i}`)

            if(v==0){
                el.innerHTML = `000`
                el.removeAttribute('style')
                el.setAttribute('style','color:black; margin: 2px;')
            } else if (v < 10) {
                el.innerHTML = `00${v}`;
                el.removeAttribute('style')
                el.setAttribute('style','color:red; margin: 2px;')
            } else if (v < 100) {
                el.innerHTML = `0${v}`;
                el.removeAttribute('style')
                el.setAttribute('style','color:red; margin: 2px;')
            } else {
                el.innerHTML = v;
                el.removeAttribute('style')
                el.setAttribute('style','color:red; margin: 2px;')
            }
        }
        //}}}
    } else if (msg.wsevent == "sitzknochenabstand") {
        // Schwerpunkte ins Bild übernehmen
        pimg.markAt (msg.schwerpunkt1_x, msg.schwerpunkt1_y, msg.schwerpunkt2_x, msg.schwerpunkt2_y);
        // Gewichtsverteilung ins Bild übernehmen
        pimg.showWeights (msg.gewicht1, msg.gewicht2);
        // Sitzknochenabstand anzeigen
        delete msg.wsevent;
        delete msg.schwerpunkt1_x;
        delete msg.schwerpunkt1_y;
        delete msg.schwerpunkt2_x;
        delete msg.schwerpunkt2_y;
	    document.getElementById("message").innerHTML = JSON.stringify(msg).replace(/</g,"&lt;").replace(/>/g,"&gt;");
    } else if (msg.wsevent == "sitzknochenanalyse") {
	    document.getElementById("message").innerHTML = JSON.stringify(msg.ellipse_links).replace(/</g,"&lt;").replace(/>/g,"&gt;");
        pimg.showEllipses (msg.ellipse_links, msg.ellipse_rechts);
    } else if (msg.wsevent == "restart") {
        console.debug ("restart");
        pimg.unmark ();
    } else {
	    document.getElementById("message").innerHTML = wsmessage.replace(/</g,"&lt;").replace(/>/g,"&gt;");
    }
} // handleWSMessage()

var conn;


/**
 * The function "connect" establishes a WebSocket connection and handles the different states of the
 * connection.
 * @returns The function `connect()` returns nothing.
 */
function connect () {//{{{
    if (window.WebSocket) {
        // Evtl. bestehende Verbindung schließen
        if (typeof conn == "object" && conn.readyState == WebSocket.OPEN) conn.close();

        // Verbindung herstellen
        try {
            // Der Server erwartet wenigstens ein (Sub)Protokoll, auch wenn das nicht benutzt wird.
            conn = new WebSocket(document.getElementById("ws_url").value, "dummy");
        } catch (error) {
            //??? nur SECURITY_ERROR
            document.getElementById("statusinfo").innerHTML = "Wenn das 'mal kein Fehler ist.";
        }
        switch (conn.readyState) {
            case WebSocket.CONNECTING: // Die Verbindung ist noch nicht hergestellt.
                document.getElementById("statusinfo").innerHTML = "WebSocket-Verbindung wird hergestellt.";
                conn.onopen = function (evt) {
                    document.getElementById("statusinfo").innerHTML = "WebSocket-Verbindung ist hergestellt.";
                    conn.onclose = function (evt) {
                        document.getElementById("statusinfo").innerHTML = "WebSocket-Verbindung wurde geschlossen.";
                        document.getElementById("message").innerHTML = "";
                    };
                };
                conn.onclose = function (evt) {
                    document.getElementById("statusinfo").innerHTML = "WebSocket-Verbindung konnte nicht hergestellt werden.";
                };
                break;
            case WebSocket.OPEN:       // Verbindung ist hergestellt und bereit darüber zu kommunizieren.
                document.getElementById("statusinfo").innerHTML = "WebSocket-Verbindung ist hergestellt.";
                break;
            case WebSocket.CLOSING:    // Verbindung ist im Prozess des Schließens.
            case WebSocket.CLOSED:     // Die Verbindung konnte nicht hergestellt werden.
                document.getElementById("statusinfo").innerHTML = "WebSocket-Verbindung konnte nicht hergestellt werden.";
                break;
        }
        // Eingetroffene Meldung vom Server bearbeiten
        conn.onmessage = function (evt) {
            handleWSMessage (evt.data);
        };
		return;
    }
	document.getElementById("statusinfo").innerHTML = "WebSocket ist in diesem Browser nicht verfügbar";
    //}}}
} // connect()

/**
 * The function checks if the WebSocket connection is open and closes it if it is.
 */
function disconnect () {
    if (typeof conn == "object" && conn.readyState == WebSocket.OPEN) {
		conn.close();
	}
}


/**
 * The function sends a message through a WebSocket connection if it is open, otherwise it displays an
 * alert message.
 */
function sendMessage () {
    if (typeof conn == "object" && conn.readyState == WebSocket.OPEN) {
        conn.send (document.getElementById("message2send").value);
    } else {
        alert ("Must connect.");
    }
}

/**
 * The clearMessage function clears the value of an input field with the id "message2send".
 */
function clearMessage() {
    document.getElementById("message2send").value = "";
}

/**
 * The function `genNormFrame` generates a grid of div elements with unique IDs based on the provided
 * number.
 * @param num - The `num` parameter is used to determine the ID of the frame element that will be
 * modified. It is an integer value that can be 0, 1, or 2.
 */
function genNormFrame(num) {
    const frame = document.getElementById(`norm${num}`)
    let lnv = num ===2 ? "i2nv" : num ===1 ? "i1nv" : num===0 ? "nv" : ""

    let html=""

    for (i_row=15; i_row>=0; --i_row) {
        i = i_row * 28;
        html += "<div style='display: flex'>";
        for (i_col=0; i_col<28; ++i_col) {
            html += `<div id="${lnv}${i}" style="margin: 2px">000</div>`;
            ++i;
        }
        html += "</div>";
    }
    frame.innerHTML=html
}

genNormFrame(0)


/***
 * Class for pressure image representation with evaluation results
 */

class PressureImage {//{{{

    constructor (canvas, size_x_cm, size_y_cm, url) {
        this.canvas = canvas;
        this.size_x_cm = size_x_cm;
        this.size_y_cm = size_y_cm;
        this.url = url;
    }

    /* X-Länge von cm in Pixel umrechnen */
    xlen2pix (len) {
        return Math.round(len/this.size_x_cm*this.canvas.width);
    }

    /* Y-Länge von cm in Pixel umrechnen */
    ylen2pix (len) {
        return Math.round(len/this.size_y_cm*this.canvas.height);
    }

    /* X-Koordinate von cm in Pixelposition umrechnen */
    xpos2pix (xpos) {
        return this.xlen2pix (xpos);
    }

    /* Y-Koordinate von cm in Pixelposition umrechnen */
    ypos2pix (ypos) {
        return this.canvas.height - this.ylen2pix (ypos) - 1;
    }

    /*
      JPEG-Bild von url holen 
      Wenn Markierungen gesetzt wurden, werden diese erneuert.
     */
    loadJPEG () {//{{{
        var img = new Image();
        var ctx = this.canvas.getContext("2d");
        var markers = this.markers;
        var weights = this.weights;
        var ellipse1 = this.ellipse1;
        var ellipse2 = this.ellipse2;
        var pimg = this;

        img.onload = function () {
            ctx.drawImage(img, 0, 0)
            pimg.showQuadrants();
            if (typeof markers === "object") {
                pimg.markAt (markers.xpos, markers.ypos, markers.xpos2, markers.ypos2);
            }
            if (typeof weights === "object") {
                pimg.showWeights();
            }
            if (typeof ellipse1 === "object") {
                pimg.drawEllipse(ellipse1);
            }
            if (typeof ellipse2 === "object") {
                pimg.drawEllipse(ellipse2);
            }
        }
        img.src = this.url + "?rnd="+(new Date).getTime();
        //}}}
    } // loadJPEG()

    /*
      Eine Ellipse zeichnen
     */
    drawEllipse (ellipse) {
        var ctx = this.canvas.getContext("2d");
        ctx.save();
        ctx.strokeStyle = "#00000F";
        ctx.beginPath();
        //ctx.arc(this.xpos2pix(ellipse.mitte.x+0.5), this.ypos2pix(ellipse.mitte.y+0.5), this.xpos2pix(ellipse.a), 0, 2*Math.PI);
        //ctx.arc(this.xpos2pix(ellipse.mitte.x+0.5), this.ypos2pix(ellipse.mitte.y+0.5), this.xpos2pix(ellipse.b), 0, 2*Math.PI);
        var w = (ellipse.w - 90) * Math.PI / 180;
        console.debug (JSON.stringify (ellipse.w)+" => "+w)
        ctx.ellipse(this.xpos2pix(ellipse.mitte.x+0.5), this.ypos2pix(ellipse.mitte.y+0.5), this.xpos2pix(ellipse.b), this.xpos2pix(ellipse.a), w, 0, 2*Math.PI);
        ctx.stroke();
        ctx.restore();
    } // drawEllipse()

    /*
      Quadranteneinteilung anzeigen
     */
    showQuadrants () {//{{{
        var ctx = this.canvas.getContext("2d");

        ctx.save();
        ctx.strokeStyle = "#FFFFFF";
        ctx.beginPath();
        ctx.moveTo (this.canvas.width/2, 0);
        ctx.lineTo (this.canvas.width/2, this.canvas.height-1);
        ctx.moveTo (0, this.canvas.height/2);
        ctx.lineTo (this.canvas.width-1, this.canvas.height/2);
        ctx.stroke();
        ctx.restore();
        //}}}
    } // showQuadrants()

    /*
      Gewichtsverteilung anzeigen
     */
    showWeights (wleft, wright) {//{{{
        if (typeof wleft !== "undefined" && typeof wleft !== "undefined") {
            this.weights = new Object();
            this.weights.left = wleft;
            this.weights.right = wright;
        } else if (typeof this.weights === "undefined") {
            return;
        }
        var wpos = this.weights.right - this.weights.left + 100; // 0 ... 200
        var pix_wpos = Math.round(this.canvas.width * wpos / 200 - 1);
        var pix_tip = Math.round(this.canvas.height / 20);
        var ctx = this.canvas.getContext("2d");
        if (pix_tip % 2 == 0)
            pix_tip += 1;
        ctx.save();
        ctx.fillStyle = "#00FF00";
        let d = new Path2D();
        d.moveTo (pix_wpos, pix_tip);
        d.lineTo (pix_wpos + Math.floor(pix_tip/2), 0);
        d.lineTo (pix_wpos - Math.floor(pix_tip/2), 0);
        d.lineTo (pix_wpos, pix_tip);
        ctx.fill(d);
        ctx.restore();
        //}}}
    }// showWeights ()

    /*
      Ellipsen anzeigen
     */
    showEllipses (ellipse1, ellipse2) {
        var pimg = this; //TODO this statt pimg benutzen ?

        // Bei der Sitzknochenanalyse liegt der Schwerpunkt ein wenig anders als bei der Ermittlung des Sitzknochenabstand
        // => korrigieren
        this.markAt (ellipse1.mitte.x, ellipse1.mitte.y, ellipse2.mitte.x, ellipse2.mitte.y);
        pimg.markAt();
        pimg.drawEllipse(ellipse1);
        pimg.drawEllipse(ellipse2);
        this.ellipse1 = ellipse1;
        this.ellipse2 = ellipse2;
    }

    /*
      Positionen markieren (Angabe in cm)
      xpos, ypos    1. Position
      xpos2, ypos2  2. Position (optional)
     */
    markAt (xpos, ypos, xpos2, ypos2) {//{{{
        var xpix = this.xpos2pix (xpos+0.5);    // Mittelpunkt der Karkierung
        var ypix = this.ypos2pix (ypos+0.5);
        var rpix = this.xlen2pix (0.5);         // Radius des Umkreises
        var dpix = Math.round (rpix / Math.sqrt(2)); // Verschiebung zu den Endpunkten des Kreuzes

        console.debug ("markAt (" + xpos + ", " + ypos + ")");
        var ctx = this.canvas.getContext("2d");
        ctx.beginPath();
        // Ausgabe als Kreuz
        ctx.moveTo (xpix-dpix, ypix-dpix);
        ctx.lineTo (xpix+dpix, ypix+dpix);
        ctx.moveTo (xpix+dpix, ypix-dpix);
        ctx.lineTo (xpix-dpix, ypix+dpix);
        ctx.stroke();

        if (typeof xpos2 === "undefined" || typeof ypos2 === "undefined")
            return;
        this.markAt (xpos2, ypos2);
        this.markers = new Object();
        this.markers.xpos = xpos;
        this.markers.ypos = ypos;
        this.markers.xpos2 = xpos2;
        this.markers.ypos2 = ypos2;
        //}}}
    }; // markAt()

    /*
      Markierung nicht mehr verwenden
      (ebenso alle anderen Auswertungsergebnisse)
     */
    unmark () {//{{{
        console.debug ("unmark");
        delete this.markers;
        delete this.weights;
        delete this.ellipse1;
        delete this.ellipse2;
        //}}}
    }// unmark()

    //}}}
} // class PressureImage 


/**
 * Test von PressureImage
 */

var pimg = new PressureImage (document.getElementById("canvas"), 28, 16, "/hockerbild");
pimg.loadJPEG ();

