# custom/kernel/ttysattel.tcl
# Kernelmodul für die Datenübernahme von einer Sattelmatte über USB
#
# Es kann nur eine Matte bedient werden.

namespace eval TTYSattel {
    variable N_COLS 16
    variable N_ROWS 28
    variable L_DATA [expr $N_COLS * $N_ROWS]
    variable fd_tty ""
    variable image_handler_procnames [list]; # Weitergabe von Druckbilddaten
    variable driver_handler_procnames [list]; # Weiterreichen von Treibermeldungen
    variable active_driver_name ""

    # Callback für anliegende Daten vom TTY-Treiber
    proc handleTTY {} {; #{{{
        variable N_COLS
        variable N_ROWS
        variable L_DATA
        variable fd_tty
        variable active_driver_name
        variable image_handler_procnames

	    set n [gets $fd_tty line]
	    if {$n < 0} {; #{{{ Fehler bei der Datenuebernahme => Abbruch
            if {[eof $fd_tty]} {
		    	if {[catch {close $fd_tty} msg]} {
	                # Die Fehlerursache wird bei close zurückgegeben.
			        srvLog [namespace current]_$fd_tty Error "$msg"
			    } else {
			        srvLog [namespace current]_$fd_tty Notice "Verbindung zum Sattel getrennt."
                    # Meldung an die clients ebenfalls durch driverChanged,
                    # falls der Treiber entfernt wurde.
                    #TODO Es könnte aber auch read_tty abgeschmiert sein.
                    # (Versuch: read_tty abschießen. Was passiert dann?
	            }
                set fd_tty ""
            }
	        return
	        #}}}
	    }

        if {$n > 0} {
            set druckbild [split $line ","]
            set l_dbld [llength $druckbild]
            if {$l_dbld == $L_DATA} {
                foreach image_handler_procname $image_handler_procnames {
                    $image_handler_procname $druckbild $N_COLS $N_ROWS
                }
            }
        }
        #}}}
    }; # proc handleTTY 


    # (Weiteren) Handler für Druckbild hinzufügen
    proc addImageHandler {cb_dbld_handler} {; #{{{
        variable image_handler_procnames

        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $image_handler_procnames $cb_dbld_handler] < 0} {
            lappend image_handler_procnames $cb_dbld_handler
            srvLog "[namespace current]::addImageHandler" Debug "$cb_dbld_handler hinzugefügt"
        }
        #}}}
    }; #proc addImageHandler 


    # Handler für Druckbild entfernen
    # @param cb_dbld_handler    Der zu entfernende Handler
    # @param force              Keine Warnung wenn der nicht gefunden wurde
    proc removeImageHandler {cb_dbld_handler {force 0}} {; #{{{
        variable image_handler_procnames

        set i [lsearch $image_handler_procnames $cb_dbld_handler]
        if {$i >=0} {
            set image_handler_procnames [lreplace $image_handler_procnames $i $i]
            srvLog [namespace current] Debug "::removeImageHandler $cb_dbld_handler entfernt"
        } elseif {!$force} {
            srvLog [namespace current] Warn "::removeImageHandler $cb_dbld_handler nicht gefunden"
        }
        #}}}
    }; # proc removeImageHandler 


    # Aktuellen TTY Status an alle Treiber Handler verteilen
    # Das kann von einer App benutzt werden, wenn ein alternativer Übertragungskanal
    # (Bluetooth) geschlossen wurde.
    proc distributeTTYState {} {; #{{{
        variable fd_tty
        variable active_driver_name
        variable driver_handler_procnames

        set msg [dict create wsevent ttystate]
        if {"$fd_tty" == ""} {
            dict set msg state inactive
        } else {
            dict set msg state active
            dict set msg driver $active_driver_name
        }

        foreach {driver_handler} $driver_handler_procnames {
            srvLog [namespace current]::distributeTTYState  Debug "$driver_handler $msg"
            $driver_handler $msg
        }
        #}}}
    }; # proc distributeTTYState 


    # (Weiteren) Handler für Treiberänderungen hinzufügen
    # Das passiert üblicherweise beim Starten einer App.
    # Dem Handler wird eine key/value-Liste mit dem wsevent "ttychange" übergeben.
    proc addDriverHandler {cb_driver_handler} {; #{{{
        variable fd_tty
        variable active_driver_name
        variable driver_handler_procnames

        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $driver_handler_procnames $cb_driver_handler] < 0} {
            lappend driver_handler_procnames $cb_driver_handler
            srvLog "[namespace current]::addDriverHandler" Debug "$cb_driver_handler hinzugefügt"
        }
        # Aktuellen Status melden
        set msg [dict create wsevent ttystate]
        if {"$fd_tty" == ""} {
            dict set msg state inactive
        } else {
            dict set msg state active
            dict set msg driver $active_driver_name
        }
        # Wahrscheinlich wird aus $msg ein Websocket-Event,
        # dessen Websocket gerade erst aktiviert wird.
        # => Verzögerung erforderlich
        after 500 "$cb_driver_handler {$msg}"
        #}}}
    }; # proc addDriverHandler 


    # Handler für Treiberänderung entfernen
    proc removeDriverHandler {cb_driver_handler} {; #{{{
        variable driver_handler_procnames

        set i [lsearch $driver_handler_procnames $cb_driver_handler]
        if {$i >=0} {
            set driver_handler_procnames [lreplace $driver_handler_procnames $i $i]
            srvLog [namespace current] Debug "::removeDriverHandler $cb_driver_handler entfernt"
        } else {
            srvLog [namespace current] Warn "::removeDriverHandler $cb_driver_handler nicht gefunden"
        }
        #}}}
    }; # proc removeDriverHandler 


    # Start der Datenübernahme vom Sattel
    # @param driver_name    Name des Treibers in /dev
    #                       (wird als id der Matte verwendet)
    # @param speed          Übertragungsgeschwindigkeit (read_tty)
    # @protocol             A ASCII oder B binär
    # @return               1 bei Erfolg, 0 bei Fehler
    proc start {driver_name speed protocol} {; #{{{
        variable fd_tty
        variable active_driver_name
        variable image_handler_procnames

        if {"$active_driver_name" != ""} {
            srvLog {TTYSattel::start} Warn "Neue Matte auf $driver_name ignoriert. Bediene bereits $active_driver_name"
            return 0
        }
        #TODO interner Fehler, wenn $fd_tty != ""
        if {[catch {
            if {"$protocol" == "A"} {
                open "|${::BINDIR}/read_tty -B $speed -m MT_SAT -n $driver_name" r
            } elseif  {"$protocol" == "B"} {
                error "Binary protocol not implemented yet."
                #TODO entsprechender Aufruf
            } else {
                error "Unknown protocol: $protocol"
            }
            } fd_tty]} {
            srvLog [namespace current]::start Error "$fd_tty"
            set fd_tty ""
            return 0
        }
        fconfigure $fd_tty -blocking 0
        fileevent $fd_tty readable [list [namespace current]::handleTTY]
        set active_driver_name $driver_name
        srvLog [namespace current]_$fd_tty Notice "Datenübernahme vom Sattel gestartet. (${::BINDIR}/read_tty, PID=[pid $fd_tty])"
        return 1
        #}}}
    }; # proc start 


    # Handler (callback) für Meldungen von watchTTYACM
    # (Gemeldet werden nur Änderungen zu Sattelmatten.)
    # @change Dict mit der von watchTTYACM::handleWatch{} gemeldeten Änderung
    #           Schlüssel:
    #               action      +|- TODO evtl. I E
    #               driver      ttyACM<nr>
    #               type        Mattentyp
    #               speed       Serielle Geschwindigkeit
    #               protocol    'A' oder 'B' für ASCII bzw. binär
    proc driverChanged {change} {; #{{{
        variable fd_tty
        variable active_driver_name
        variable driver_handler_procnames

        srvLog "[namespace current]" Debug "driverChanged $change"
        set driver [dict get $change "driver"]
        # wsevent wird zur Erhaltung der Kompatibilität zunächst wie in V2 gesetzt.
        switch [dict get $change "action"] {
            + {
                # Meldung an die Clients
                set msg [list wsevent plugged driver $driver type [dict get $change type]]
                # Datenübernahme starten
                if {[start $driver [dict get $change "speed"] [dict get $change "protocol"]]} {
                    set active_driver_name $driver
                    lappend msg start ok
                } else {; # Fehler
                    lappend msg start error
                }
            }
            - {
                if {"$active_driver_name" != $driver} {
                    srvLog [namespace current] Notice "Zuvor nicht registrierter Treiber $driver entfernt."
                    return
                }
                if {"$fd_tty" != ""} {; # handleTTY hat Datenübernahme nicht gestoppet.
                    catch {close $fd_tty}
                    set fd_tty ""
                }
                set active_driver_name ""
                # Meldung an die Clients
                set msg [list wsevent unplugged driver $driver type [dict get $change type]]
                #TODO Bedeutung bei Sattelmatte unklar:  MATTE 0
            }
            default {
                return
            }
        }; # switch action
        #TODO Was machen wir damit:
        # action E: (Error)
        # action I: (Info) ?
        srvLog [namespace current] Debug "$msg"
        # Meldung an die WS-Clients als JSON-Objekt abschicken.
        # Das bleibt vorerst wegen der Kompatibilität.
        ::WSServer::disposeServerMessage messages text [::kvlist2json $msg]
        # Callbacks aufrufen
        # Dazu zunächst Schlüssel ein wenig umbauen
        set msg [dict merge $msg [dict create wsevent "ttychange" change [dict get $msg wsevent]]]
        foreach {driver_handler} $driver_handler_procnames {
            srvLog [namespace current] Debug "$driver_handler $msg"
            $driver_handler $msg
        }
        #}}}
    }; # proc driverChanged 


    # Initialisierung nach Laden des Moduls
    proc init {} {; #{{{
        # Anmelden bei watchTTYACM
        ::kernel::watchTTYACM::addDriverchangedHandler [namespace current]::driverChanged MT_SAT*
        #}}}
    }; # proc init 

}; # namespace eval TTYSattel 

set mod_loaded TTYSattel

