# custom/kernel/btsattel.tcl
# Kernelmodul für die Datenübernahme von einem Sattel über Bluetooth.
# Startet bin/btsattel.py und steht bereit für den Aufruf von Kommandoprozeduren.
#
# Es kann nur eine Matte bedient werden.

#TODO Fehlernummer bei (connection) fail

namespace eval BTSattel {
    variable N_COLS 16
    variable N_ROWS 28
    variable L_DATA [expr $N_COLS * $N_ROWS]
    variable TRY_CONNECT_TIMEOUT 20000
    variable fd_bt ""
    variable data_handler_procnames [list]; # Weitergabe von Druckbilddaten
    variable connection_handler_procnames [list]; # Weitergabe von Scanergebnissen und Verbindungsabbruch
                                                  # Der 1. Handler in der Liste ist der interne.
    variable trying_connect 0;          # Es läuft gerade ein Verbindungsversuch.
    variable trying_connect_device "";  # Erwartet wird dieser Controller. (BT-Name)
                                        # oder ein beliebiger, wenn ""
    variable trying_connect_timeout_id "";  # ... zum Zurücksetzen

    # Timeout für tryConnect{}
    proc tryConnectTimeout {} {; #{{{
        variable trying_connect
        variable connection_handler_procnames

        set trying_connect 0
        srvLog [namespace current] Warn "tryConnect timeout."
        # Fehler (nur) an die externen Connection Handler senden
        foreach connection_handler [lrange $connection_handler_procnames 1 end] {
            $connection_handler "{\"btevent\": \"error\", \"class\": \"fail\", \"msg\": \"tryConnect timeout.\"}"
        }
        #}}}
    }; # proc tryConnectTimeout 


    # (Interner) Connection Handler
    # Behandelt nur die Statusereignisse während trying_connect.
    proc handleConnectionChange {json_change} {; #{{{
        variable trying_connect
        variable trying_connect_device
        variable trying_connect_timeout_id
        variable TRY_CONNECT_TIMEOUT
        variable fd_bt
        variable connection_handler_procnames

        if {$trying_connect} {
            set dict_change [::json::json2dict $json_change]
            set btevent [dict get $dict_change "btevent"]

            if {"$btevent" == "status"} {; #{{{
                set devices [dict get $dict_change "devices"]
                switch [llength $devices] {
                    0 {; # Keinen Controller gefunden => Fehler
                        set trying_connect 0
                        # Fehler (nur) an die externen Connection Handler senden
                        foreach connection_handler [lrange $connection_handler_procnames 1 end] {
                            $connection_handler "{\"btevent\": \"error\", \"class\": \"fail\", \"msg\": \"No devices found.\"}"
                        }
                    }
                    1 {; # Genau einen Controller gefunden. Super! notify versuchen.
                        set device_name [dict get [lindex $devices 0] name]
                        srvLog [namespace current] Debug "One device found ($device_name) => try connect"
                        # Falls $trying_connect_device angegeben, muß der Name übereinstimmen.
                        if {"$trying_connect_device"=="" || "$trying_connect_device"=="$device_name"} {
                            puts $fd_bt "notify $device_name"
                            flush $fd_bt
                            srvLog [namespace current] Debug "Notify $device_name started."
                            # Es gibt hier keine Rückmeldung über Erfolg oder Mißerfolg.
                            # (Entweder kommen später Daten oder eben nicht.)
                            # trying_connect wird beim Eintreffen der ersten Daten zurückgesetzt.
                            # Wenn keine Daten kommen, ist das aufgrund eines Fehlers,
                            # der auch gemeldet wird. Dann wird trying_connect ebenfalls zurückgesetzt.
                            # Sicherheitshalber wird noch ein timeout gesetzt.
                            set trying_connect_timeout_id [after $TRY_CONNECT_TIMEOUT [namespace current]::tryConnectTimeout]
                        } else {
                            # Fehler (nur) an die externen Connection Handler senden
                            foreach connection_handler [lrange $connection_handler_procnames 1 end] {
                                $connection_handler "{\"btevent\": \"error\", \"class\": \"fail\", \"msg\": \"Expected device ($trying_connect_device) not found.\"}"
                            }
                            set trying_connect 0
                        }
                    }
                    default {; # Mehrere Controller gefunden. => Einer davon ist auszuwählen.
                        set json_devices ""
                        foreach device $devices {
                            if {[string length $json_devices] > 0} {
                                append json_devices ", "
                            }
                            set device_name [dict get $device name]
                            append json_devices "\"$device_name\""
                            if {"$trying_connect_device" == $device_name} {
                                puts $fd_bt "notify $device_name"
                                flush $fd_bt
                                break
                            } else {
                                set device_name ""
                            }
                        }
                        if {$device_name == ""} {; # nichts Passendes gefunden
                            # => Mehrdeutigkeit an die externen Connection Handler senden.
                            foreach connection_handler [lrange $connection_handler_procnames 1 end] {
                                $connection_handler "{\"btevent\": \"ambiguity\", \"devices\": \[$json_devices\]}"
                            }
                            set trying_connect 0
                        }
                    }
                }
                flush stdout
                #}}} if $btevent == "status"
            } elseif {"$btevent" == "error"} {
                srvLog [namespace current] Warn "tryConnect failed: $json_change"
                if {"$trying_connect_timeout_id" != ""} {; # Timeout zurücksetzen
                    after cancel $trying_connect_timeout_id
                    set trying_connect_timeout_id ""
                }
                set trying_connect 0
            } elseif {"$btevent" == "data"} {
                srvLog [namespace current] Debug "tryConnect successful: $json_change"
                if {"$trying_connect_timeout_id" != ""} {; # Timeout zurücksetzen
                    after cancel $trying_connect_timeout_id
                    set trying_connect_timeout_id ""
                }
                set trying_connect 0
            }
        }; # if $trying_connect
        #}}}
    }; # proc handleConnectionChange 

    lappend connection_handler_procnames [namespace current]::handleConnectionChange 

    # Daten vom Bluetooth Agent liegen an.
    # Das kann eine Statusmeldung oder Daten sein.
    # => Eine Zeile holen und verarbeiten
    proc btRead {} {; #{{{
        variable N_COLS
        variable N_ROWS
        variable L_DATA
        variable fd_bt
        variable connection_handler_procnames
        variable data_handler_procnames
        variable trying_connect

        set n [gets $fd_bt line]
        if {$n > 0} {; # Daten sind angekommen.
            #??? srvLog [namespace current]::btRead Debug "$line"
            if {[string match {\[*\]} $line]} {; # Statusarray
                # ... in Objekt verpacken
                set line "{\"btevent\": \"status\", \"devices\": $line}"
            }
            if {[string match {{*}} $line]} {; # btevent
                # An die Connection Handler verteilen
                #??? srvLog [namespace current]::btRead Debug "Dispose: $connection_handler_procnames"
                foreach connection_handler $connection_handler_procnames {
                    $connection_handler $line
                }
            } elseif {[string match {[0-9]*} $line]} {; # Druckdaten
                set druckbild [split $line ","]
                set l_dbld [llength $druckbild]
                if {$l_dbld == $L_DATA} {
                    foreach data_handler $data_handler_procnames {
                        $data_handler $druckbild $N_COLS $N_ROWS
                    }
                }
                if {$trying_connect} {
                    # An alle Connection Handler verteilen.
                    # Der interne wird trying_connect zurücksetzen.
                    # Die externen können durch Hinzufügen eines Datenhandlers
                    # die Datenübernahme beginnen.
                    foreach connection_handler $connection_handler_procnames {
                        $connection_handler "{\"btevent\": \"data\"}"
                    }
                }
            }
            # else Datenmüll, der ignoriert wird
        } elseif {$n < 0} {
            srvLog $fd_bt Error "Error reading from btagent"
            catch {close $fd_bt}
            srvLog [namespace current]::btRead Info "Connection ($fd_bt) to btagent closed."
        } elseif {[eof $fd_bt]} {
            srvLog $fd_bt Error "EOF from btagent received."
            catch {close $fd_bt}
            srvLog [Namespace current] Info "Connection ($fd_bt) to btagent closed."
        }
        # else $n == 0
        #}}}
    }; # proc btRead 


    # (Weiteren) Handler für Verbindungsänderungen hinzufügen
    # Dem Handler wird ein JSON-String mit dem Schlüssel "btevent" übergeben.
    # Abhängig vom zugehörigen Wert kann es weitere Schlüssel geben.
    # @param cb_connection_handler  Name des Callbacks (mit komplettem Namespace)
    proc addConnectionHandler {cb_connection_handler} {; #{{{
        variable connection_handler_procnames

        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $connection_handler_procnames $cb_connection_handler] < 0} {
            lappend connection_handler_procnames $cb_connection_handler
            srvLog "[namespace current]::addConncetionHandler" Debug "$cb_connection_handler hinzugefügt"
        }
        #}}}
    }; # proc addConnectionHandler 


    # Handler für Verbindungsänderungen entfernen
    # @param cb_connection_handler  Name des Callbacks (mit komplettem Namespace)
    proc removeConnectionHandler {cb_connection_handler} {; #{{{
        variable connection_handler_procnames

        set i [lsearch $connection_handler_procnames $cb_connection_handler]
        switch $i {
            0 {
                srvLog [namespace current] Warn "::removeConnectionHandler $cb_connection_handler not found"
            }
            1 {
                # => Interner Handler, darf nicht entfernt werden!
                srvLog [namespace current] Warn "::removeConnectionHandler $cb_connection_handler must not be removed"
            }
            default {
                set connection_handler_procnames [lreplace $connection_handler_procnames $i $i]
                srvLog [namespace current] Debug "::removeConnectionHandler $cb_connection_handler removed"
            }
        }
        #}}}
    }; # proc removeConnectionHandler 


    # (Weiteren) Handler für Druckbilddaten hinzufügen
    # Dem Handler werden die Arumente values (Liste) n_cols n_rows übergeben.
    # @param cb_data_handler  Name des Callbacks (mit komplettem Namespace)
    proc addDataHandler {cb_data_handler} {; #{{{
        variable data_handler_procnames
        variable N_COLS
        variable N_ROWS
        variable L_DATA

        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $data_handler_procnames $cb_data_handler] < 0} {
            lappend data_handler_procnames $cb_data_handler
            srvLog "[namespace current]::addDataHandler" Debug "$cb_data_handler hinzugefügt"
            # Zwecks Initialisierung ein leeres Druckbild übergeben
            $cb_data_handler [lrepeat $L_DATA 0] $N_COLS $N_ROWS
        }
        #}}}
    }; # proc addDataHandler 


    # Handler für Druckbilddaten entfernen
    # @param cb_data_handler    Name des Callbacks (mit komplettem Namespace)
    # @param force              Keine Warnung, wenn der nicht gefunden wurde
    proc removeDataHandler {cb_data_handler {force 0}} {; #{{{
        variable data_handler_procnames

        set i [lsearch $data_handler_procnames $cb_data_handler]
        if {$i >=0} {
            set data_handler_procnames [lreplace $data_handler_procnames $i $i]
            srvLog [namespace current] Debug "::removeDataHandler $cb_data_handler entfernt"
        } elseif {!$force} {
            srvLog [namespace current] Warn "::removeDataHandler $cb_data_handler nicht gefunden"
        }
        #}}}
    }; # proc removeDataHandler 


    # Controller aus dem Bluetooth System entfernen
    # Fehlermeldungen werden ins Logfile geschrieben.
    # @param device_name    Nur angegebenes Geräte entfernen (sonst alle SmarCover)
    # @return   0 bei Fehler, 1 sonst
    proc removeDevices {{device_name {}} } {; #{{{
        srvLog [namespace current]::removeDevices Debug "exec $::dir/bin/btremove.sh $device_name"
        if {[catch {exec $::dir/bin/btremove.sh $device_name} msg]} {
            srvLog [namespace current]::removeDevices Error $msg
            return 0
        }
        return 1
        #}}}
    }; # proc removeDevices 


    # Versuche eine Bluetooth Verbindung herzustellen
    # Der Vorgang wird hier mit "scan" nur angeschoben und beim Eintreffen des Ergebnisses
    # beim internen connection handler (proc handleConnectionChange) fortgesetzt,
    # der das Weitere veranlasst:
    # 1. Wenn kein Controller gefunden wurde, wird eine entsprechende Fehlermeldung
    #    an die registrierten connection_handler verteilt.
    # 2. Bei genau einem gefundenen Controller wird ...
    # 2.a.  ... bei nicht angegebener device_name die Verbindung mit diesem hergestellt.
    # 2.b.  ... bei angegebener device_name erst nach Namensvergleich mit dem Verbinden bzw.
    #       mit einer Fehlermeldung fortgefahren.
    # 3.    Bei mehreren gefunden Controllern ...
    # 3.a.  ... und angegeber device_name wird die Verbindung hergestellt, sofern der richtige dabei ist.
    # 3.b.  Anderfalls werden die Controllernamen zur Auswahl des richtigen
    #       an die Connection Handler verteilt.
    # @param device_name    Bluetooth-Name des Controllers
    proc tryConnect {{device_name {}} } {; #{{{
        variable fd_bt
        variable trying_connect_device
        variable trying_connect

        if {$trying_connect} {
            srvLog [namespace current] Warn "tryConnect while already trying connect ignored."
            return
        }
        #TDOD "$fd_bt" == "" => Fehler Bluetooth nicht verfügbar (evtl. nicht installiert.)
        srvLog [namespace current] Debug "Starting scan ..."
        puts $fd_bt "stop"; # ... falls btagent nach unerwartetem Verbindungsende
                            # in der inneren Kommandoschleife steckengeblieben ist.
        puts $fd_bt "scan"
        set trying_connect_device $device_name
        set trying_connect 1
        flush $fd_bt
        srvLog [namespace current] Debug "Scan startet."
        #}}}
    }; # proc tryConnect 


    # Versionsabfrage an den btagent senden.
    # Die Antwort kommt als Connection Event.
    proc agentVersion {} {; #{{{
        variable fd_bt

        puts $fd_bt "version"
        flush $fd_bt
        srvLog [namespace current] Debug "Query agent version."
        #}}}
    }; # proc agentVersion 


    # "stop" an btagent senden.
    # Wenn keine Verbindung besteht, läuft das ins Leere.
    proc disconnect {} {; #{{{
        variable fd_bt
        variable trying_connect

        # Läuft gerade ein Verbindungsversuch ?
        if {$trying_connect} {
            # Fehler (nur) an die externen Connection Handler senden
            foreach connection_handler [lrange $connection_handler_procnames 1 end] {
                $connection_handler "{\"btevent\": \"error\", \"class\": \"client\", \"msg\": \"disconnect while trying connect.\"}"
            }
            return
        }
        puts $fd_bt "stop"
        flush $fd_bt
        #}}}
    }; # proc disconnect 


    # Prüfe, ob btagent.py gestartet wurde
    # @return 1 gestartet, 0 sonst
    proc btagentStarted {} {; #{{{
        variable fd_bt

        return [expr {"$fd_bt" != ""}]
    #}}}
    }; # proc btagentStarted 


    # btagent.py starten, sofern python bleak verfügbar ist
    # @param fd_piplist Filedescr. für "pip3 list"
    proc startBtagent {fd_piplist} {; #{{{
        variable fd_bt

        set count [string trim [read $fd_piplist]]
        # Bei count == 0 endet grep mit status 1, was hier Exception auslöst.
        catch {close $fd_piplist}
        if {$count == 0} {
            srvLog [namespace current]::startBtagent Warn "Bluetooth ist nicht verfügbar. (Python3 bleak ist nicht installiert.)"
            return
        }
        # Den Bluetooth Agent starten
        set btagent_py [file join {*}[lrange [file split $::BINDIR] 0 end-1] btagent.py]
        if {[file executable $btagent_py]} {; # starten
            srvLog [namespace current] Debug "$btagent_py wird gestartet ..."
            if {[catch {
                # -d optional entsprechend aktuellem Debuglevel
                set opt_debug ""
                if {[string compare -length 5 "debug" $::LOGLEVEL] == 0} {
                    set opt_debug " -d"
                }
                open "|$btagent_py$opt_debug -n" r+
                } fd_bt]} {
                srvLog [namespace current]::startBtagent Error "$fd_bt"
                set fd_bt ""
            } else {
                fconfigure $fd_bt -blocking 0 -buffering line
                fileevent $fd_bt readable [namespace current]::btRead
                srvLog [namespace current]::startBtagent Info "$btagent_py wurde gestartet. ($fd_bt)"
            }
        } else {; # Fehlermeldung ins Log
            srvLog [namespace current]::startBtagent Error "$btagent_py nicht vorhanden oder nicht ausführbar."
        }
        #}}}
    }; # proc startBtagent 


    # Initialisierung nach Laden des Moduls
    proc init {} {; #{{{
        # Prüfen, ob Bluetooth vorhanden ist
        # 1. python3-pip ?
        if {[catch {exec dpkg -l python3-pip > /dev/null}]} {
            srvLog [namespace current] Warn "Bluetooth ist offenbar nicht verfügbar. (Python3 pip ist nicht installiert.)"
            return
        }
        # 2. python bleak ?
        # Die Abfrage mit pip3 list verzögert den Startvorgang über Gebühr.
        # Sie wird deshalb in den Hintergrund verlagert.
        if {[catch {
            set fd_piplist [open "|pip list 2>/dev/null | grep -c bleak" "r"]
            fileevent $fd_piplist readable "[namespace current]::startBtagent $fd_piplist"
            } errmsg]} {
            srvLog [namespace current]::init Error "$errmsg"
        }
        #}}}
    }; # proc init 

}; # namespace eval BTSattel 

set mod_loaded BTSattel

