#!/bin/bash
#
# .../setup/setup.sh
#
# Velometrik GmbH
#
# Installationsskript für vmkstationd
# Dieses Skript ist der Startpunkt für eine Neuinstallation.
# Das Logfile ist /var/local/log/vmksdsetup.log.
#
# Historie:
#   24.02.2023 Siegmar Müller   (Neu) Begonnen
#   28.02.2023 Siegmar Müller   Fertig
#   20.11.2023 Siegmar Müller	Setup bluetooth ausschaltbar
#   22.11.2023 Siegmar Müller	Demostreams eingebaut
#   28.11.2023 Siegmar Müller	Aufruf von update-rc.d und service mit Pfad
#


LOGFILE=/var/local/log/vmksdsetup.log
SETUP_BT=true

# Komponenten vom Setup aussschließem
# Aufruf: exclude <componenets>
#           <components>: BT
exclude() {
    local COMPONENTS=`echo $1 | sed 's/,/ /g'`
    for COMPONENT in $COMPONENTS; do
        case $COMPONENT in
            BT) SETUP_BT=false;;
            *) echo "Unbekannte Komonente zum Ausschließen: $COMPONENT"; exit 1;;
        esac
    done
}


# Hilfe ausgeben
help() {
    echo
    echo "vmkstationd Initiales Setup:"
    echo
    echo "Aufruf: $0 [<optionen>]"
    echo "Optionen:"
    echo "-x <components>   Komponenten vom Setup ausschließen"
    echo "                  <components> Liste mit Komma als Trenner"
    echo "                  Komponenten:    BT BlueTooth"
    echo "-l <logfile>      Logdatei ($LOGFILE)"
    echo "-?                Hilfe ausgeben und beenden"
    if [ $UID -ne 0 ]; then
        echo "(Die Ausführung erfordert root-Rechte.)"
    fi
    echo
}

# Start
echo
echo "$0 $@"
echo

# Optionen aus der Kommandozeile holen.
while getopts x:l:? OPTION; do
    case $OPTION in
        x) exclude $OPTARG;;
        l) LOGFILE=$OPTARG;;
        ?) help; exit 0;;
    esac
done

# Ausführung als root
if [ $UID -ne 0 ]; then
    echo "Das Skript benötigt 'root' Rechte."
    su -c "$0 $*"
    exit $?
fi

SETUP="$(cd $(dirname $0) > /dev/null 2>&1; pwd -P)/"
VMKSTATIOND=`echo $SETUP | sed 's/setup\///'`
# In dem Verzeichnis muß vmkstationd.tcl sein:
if [ ! -x ${VMKSTATIOND}vmkstationd.tcl ]; then
    echo "$0 befindet sich nicht in der Umgebung von vmkstationd3."
    exit 1
fi

### Vorbereitung für Logausgaben und Fehlertest mit eventuellem Abbruch

if [ -f $LOGFILE ]; then
    echo -e "\n-------------------------------------------------------------------------------" >> $LOGFILE
else
    # Das Logverzeichnis existiert noch nicht in einem neu aufgesetzten System.
    mkdir -p `dirname $LOGFILE`
fi
echo "setup.sh `date '+%d.%m.%Y %H:%M:%S'`" | tee -a $LOGFILE
SETUP_DIR=`echo \`realpath $0\` | sed 's?/setup.sh??'`
echo "SETUP_DIR = $SETUP_DIR" | tee -a $LOGFILE

# Ausgabe von Logmeldungen in der Form:
#   echo "Meldung" | tee -a $LOGFILE
log() {
    echo "$*" | tee -a $LOGFILE
}


# Setup bei Fehler abbrechen
#   cancelSetup [<exit_status>]
# Ohne Angabe von <exit_status> wird mit Status 0 abgebrochen.
# Bei <exit_status> 0 geht es weiter.
cancelSetup() {
    if [ $# -eq 0 ]; then
        log "setup.sh abgebrochen."
        exit 0
    fi
    if [ $1 -ne 0 ]; then
    	log "setup.sh mit Status $1 abgebrochen"
    	exit $1
    fi
} # cancelSetup()


# Umleitung von Kommandoausgaben:
#   <command> 2>&1 | tee -a $LOGFILE
# Umleitung von Kommandoausgabe und Statusauswertung:
#   <command> 2>&1 | tee -a $LOGFILE
#   [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1


# Läuft bereits ein vmkstationd.sh ?
service vmkstationd.sh status 2>&1 | tee -a $LOGFILE
RC=${PIPESTATUS[0]}
if [ $RC -eq 0 ]; then # Service läuft
	log "vmkstationd wird gestoppt und neu installiert."
    service vmkstationd.sh stop 2>&1  | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1
elif [ $RC -eq 3 ]; then # Service existiert, läuft aber nicht
    log "vmkstationd wird neu installiert."
elif [ $RC -eq 4 ]; then # Service existiert noch nicht
    log "vmkstationd wird erstmalig installiert."
fi
if [ $RC -ne 4 ]; then
    log "WARNUNG: Dieses Setup kann nicht auf eine eventuelle Vorversion zurücksetzen."
#   Das behindert das automatische Setup mit vlbupdate:
#    echo "Dieses Setup kann nicht auf eine eventuelle Vorversion zurücksetzen."
#    read -p "Fortfahren? [Yn]" A
#    A=${A:-Y}
#    if [ ${A^^} != "Y" ]; then
#        log "Setup vom Bediener abgebrochen."
#        cancelSetup
#    fi
fi

# Debian-Pakete, die der Stationsdaemon benötigt.
DEBIAN_PAKETE="tcl8.6 tcllib tcl-thread tcl-tls tcl8.6-tdbc tcl8.6-tdbc-sqlite3"
if `$SETUP_BT`; then
    # Bluetooth erfordert zusätzliche Python Pakete
    DEBIAN_PAKETE="$DEBIAN_PAKETE python3-pip"
fi
for DEB_PAKET in $DEBIAN_PAKETE; do
    STATUS=`dpkg-query --showformat '${db:Status-Status}' -W $DEB_PAKET 2>&1`
    RC=$?
    if [ $RC -eq 2 ]; then
        log "FEHLER: $STATUS"
        cancelSetup 2
    fi
    if [ $RC -eq 1 ]; then
        STATUS="not-installed"
    fi
    if [ "$STATUS" == "installed" ]; then
        log "$DEB_PAKET ist bereits installiert."
    elif [ "$STATUS" == "not-installed" ]; then
        apt-get -y install $DEB_PAKET 2>&1 | tee -a $LOGFILE
    	[ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    else
        log "FEHLER: Unbekannter Status ($STATUS) für Paket $DEB_PAKET."
        cancelSetup 1
    fi

done

if `$SETUP_BT`; then
    # Python bleak installieren
    OUT=`pip3 list | grep bleak`
    if [ "$OUT" == "" ]; then
        log "Python bleak muß installiert werden."
        pip3 install bleak 2>&1 | tee -a $LOGFILE
        if [ $? -ne 0 ]; then
            log "bleak konnte nicht installiert werden."
            cancelSetup 1
        fi
        log "bleak wurde installiert."
    else
        log "Bereits installiert: $OUT"
    fi
else
    log "Setup von Python bleak wurde deaktiviert."
fi


# /usr/local/etc/vmkstationd.conf
cp -v $SETUP/vmkstationd.conf /usr/local/etc/ 2>&1  | tee -a $LOGFILE
[ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1
chgrp -v staff /usr/local/etc/vmkstationd.conf 2>&1 | tee -a $LOGFILE
[ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1
chmod -v g+w /usr/local/etc/vmkstationd.conf 2>&1 | tee -a $LOGFILE
[ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1


# dhcp-exit-hook
cp -v $SETUP/dhcpcd.exit-hook /etc/dhcpcd.exit-hook 2>&1 | tee -a $LOGFILE
[ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1

# Link /usr/local/vmkstationd_next erstellen
rm -f /usr/local/vmkstationd_next 2>&1 | tee -a $LOGFILE
[ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1
ln -s ${VMKSTATIOND} /usr/local/vmkstationd_next 2>&1 | tee -a $LOGFILE
[ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1
log "Link /usr/local/vmkstationd_next auf ${VMKSTATIOND} erstellt."


# Binaries prüfen
log "Prüfe Auflösung von shared libraries"
for BIN in `ls ${VMKSTATIOND}/bin/$(uname -m)/*`; do
    if `ldd $BIN | tee -a $LOGFILE | grep -q 'not found'`; then
        log "Shared Libraries für $BIN nicht auflösbar"
        cancelSetup 1
    else
        log "$BIN O.K."
    fi
done
for LIB in `ls ${VMKSTATIOND}/lib/$(uname -m)/lib*.so`; do
    if `ldd $LIB | tee -a $LOGFILE | grep -q 'not found'`; then
        log "Shared Libraries für $LIB nicht auflösbar. Prüfe Alternativen ..."
        FOUND=false
        for LIB2 in `ls ${VMKSTATIOND}/lib/$(uname -m)/lib*.so.[0-9]*`; do
            if `ldd $LIB2 | tee -a $LOGFILE | grep -q 'not found'`; then
                :
            else
                cp -v $LIB2 $LIB | tee -a $LOGFILE
                log "$LIB2 wird als Alternative für $LIB verwendet."
                FOUND=true
                break
            fi
        
        done
        $FOUND || { log "Keine Alternative gefunden."; cancelSetup 1; }
    else
        log "$LIB O.K."
    fi
done

# Demostreams entpacken
tar xf $SETUP/rawstreams.tar.gz -C / | tee -a $LOGFILE
[ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1

# Startskript kopieren und scharf schalten
cp -v $SETUP/vmkstationd.sh /etc/init.d/ 2>&1  | tee -a $LOGFILE
[ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1
/sbin/update-rc.d vmkstationd.sh defaults 2>&1  | tee -a $LOGFILE
[ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1
log "Service vmkstationd.sh wird gestartet ..."
/sbin/service vmkstationd.sh start 2>&1  | tee -a $LOGFILE
[ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1

log "Setup vmkstationsd.sh erfolgreich ausgeführt."

