#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""

File: btagent.py

Velometrik GmbH

Vermittler zwischen Bluetooth und vmkstationd3

Voraussetzungen:
pip3 install bleak

Allgemeines:
Der Bluetoothagent vermittelt die Kommunikation zwischen einem Velometrik-Controller
und dem Velometrik Stationsdaemon (vmkstationd3).
Die Kommunikation zwischen Daemon und Agent erfolgt ausschließlich über stdin und stdout.
Über stdin werden Kommandos gelesen, über stdout werden die Druckdaten
(durch Komma getrennte ASCII-Zahlen, ein Druckbild je Zeile) oder Meldungen (JSON) ausgegeben.
Es gibt eine äußere und eine innere Kommandoschleife.
In der äußeren Kommandoschleife werden nur Kommandos erwartet aber keine Druckdaten ausgegeben.
Zur Ausgabe von Druckdaten (angeschoben durch das Kommando 'notify') geht der Agent
in die innere Kommandoschleife, die nur Kommandos zum Beenden der Druckdatenweitergabe akzeptiert.
Die zurückgegebenen JSON-Strings repräsentieren entweder Statusinformationen
zu den erkannten Controllern oder Meldungen. Im ersten Fall (Statusinformationen)
handelt es sich um JSON-Arrays (erstes Zeichen '[', letztes Zeichen ']'),
im zweiten Fall (Meldungen) sind es JSON-Objekte (erstes Zeichen '{', letztes Zeichen '}').

Die Kommunikation beginnt mit dem Kommando 'scan', das die erkannten Geräte liefert.
Mit dem Kommando 'notify' wird danach die Datenübernahme gestartet.
Mit dem Kommando 'stop' wird sie beendet.

Die Fehlermeldungen sind je einer Fehlerklasse zugeordnet:
"disconnected"  Unerwarteter Verbindungsabbruck
"fail"          Verbindungsaufbau gescheitert
"comm"          Sonstiger Kommunkationsfehler
"internal"      Interner Fehler, sollte nie auftreten
"client"        Fehler vom Client, falsches Kommando

Solange eine Verbindung zum Controller besteht, blinkt dieser blau.

Kommandos:
s. Hilfe (-?)

Hinweise:
    Wenn bei scan mehr als ein nicht gepairtes Gerät eingeschaltet ist,
    werden mehrere Geräte mit demselben Namen ('SmartCov') gefunden.
    Bei Auswahl eines beliebigen davon erkennt man am Blinken der Kontrolleuchte
    welches es war. Sinnvoller ist evtl. die Aufforderung, nur ein Gerät einzuschalten.

    Bei einem Kommunikationsfehler ist die Standardreaktion, nacheinander stop, scan und notify auszuführen

Historie:
    25.07.2022  Siegmar Müller  Umbau aus Testskript scanpairnotify.py
    29.07.2022  Siegmar Müller  Version 1.0.0
    04.08.2022  Siegmar Müller  Korrekturen => Version 1.0.1
    05.08.2022  Siegmar Müller  Rückgabe der Klasse bei unbekannten Exceptions => Version 1.0.2
    06.08.2022  Siegmar Müller  Neues btevent "disconnected" => Version 1.0.3
    12.08.2022  Siegmar Müller  TODO => Version 1.0.4
    26.01.2023  David Sigler    Version 1.0.5 -> Handle Devices with no name string in advertisement data (BLE beacons)
    03.02.2023  Siegmar Müller  Fehlendes 'flush' ergänzt => Version 1.0.6

"""

import asyncio
from bleak import BleakScanner, BleakClient, BleakError
import sys
import json

VERSION = "1.0.6"
# Das ist die Service-ID des Controllers zum Lesen der Daten.
# (Senden aus Controllersicht)
UART_TX_CHAR_UUID = "a73e9a10-628f-4494-a099-12efaf72258f"

opt_not_null = False # 0-Datenblöcke unterdrücken (Option -n)
debug = False   # Option -d

# Kommandozeile nach Option -n durchforsten
for i in range(1, len(sys.argv)):
    if sys.argv[i] == "-n":
        opt_not_null = True
    elif sys.argv[i] == "-d":
        debug = True
    elif sys.argv[i] == "-?":
        print (f"Bluetooth agent for Velometrik controller V{VERSION}")
        print (f"{sys.argv[0]} [<options>]")
        print ("Options:")
        print ("\t-n Don't pass zero images")
        print ("\t-d Show debug messages")
        print ("\t-? Show this help and exit")
        print ("Commands (outer commandloop):")
        print ("\tscan\t\tScan for bluetooth devices named 'SmartCov*'")
        print ("\tstatus\t\tShow last scan result")
        print ("\tnotify <name>\tStart reading fron device <name> and enter inner commandloop")
        print ("\tset nonulls on|off\tPass/don't pass zero images")
        print ("\tstop\t\tDummy command")
        print ("\tversion\t\tQuery version")
        print ("\tquit\t\tStop reading if reading and exit")
        print ("Commands (inner commandloop):")
        print ("\tstop\t\tStop reading and exit inner commandloop")
        print ("\tquit\t\tStop reading and exit this program")
        quit()

# Ausgabe einer Debugmeldung, falls gewünscht
if debug:
    def print_dbg (msg):
        json.dump({'btevent': 'debug', 'msg': msg}, sys.stdout); print("", flush=True)
else:
    def print_dbg (msg):
        pass


# Ausgabe einer Fehlermeldung
def print_err (errclass, msg, nr=0):
    if nr == 0: # Keine Fehlernummer
        json.dump({'btevent': 'error', 'class': errclass, 'msg': msg}, sys.stdout); print("", flush=True)
    else:
        json.dump({'btevent': 'error', 'class': errclass, 'nr': nr, 'msg': msg}, sys.stdout); print("", flush=True)


# Ausgabe eines beliebigen Events
def print_btevent (btevent, msg):
    json.dump({'btevent': btevent, 'msg': msg}, sys.stdout); print("", flush=True)


devices = []    # (list) Wie von BleakScanner gefunden
devinfos = []   # ... aus devices extrahiert
                # (Dictionaries mit 'name', 'address', 'paired', 'connected')

# Daten für die Verarbeitung eines Datenblocks initialisieren
before = '?'
i_line = -1
commas = 0
dtaline = ''
block1 = True   # Erster empfangener Datenblock
dtavalid = True
lines = []      # Puffer für die Zeilen eines Datenbtblocks initialisieren


# (Neu)initialisierung der Lesedaten
# (Aufruf vor start_notify())
def init_data (): #{{{
    global before
    global i_line
    global commas
    global dtaline
    global block1
    global dtavalid
    global lines

    before = '?'
    i_line = -1
    commas = 0
    dtaline = ''
    block1 = True
    dtavalid = True
    lines = []
    for i in range(27):
        lines.append('0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,')
    lines.append('0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0')
#}}} init_data()


# Callback-Routine für angekommene Daten
def data_received (sender, data): #{{{
    global opt_not_null
    global before
    global i_line
    global commas
    global dtaline
    global block1
    global dtavalid
    global lines

    btblock = data.decode()
    for char in btblock:
        if (char == '\n'): # Ende der Zeile
            if before == '\n': # Ende des Datenblocks
                if opt_not_null: # Nicht 0 Werte suchen
                    print_block = False
                    for line in lines:
                        if len(line.strip("0,")) > 0:
                            print_block = True
                            break
                else: # immer ausgeben
                    print_block = True
                if print_block or block1: # Vorhergehenden Block ausgeben
                    for line in lines:
                        print(line, end='')
                    print('', flush=True)
                # Initialisieren für nächsten Datenblock
                block1 = False
                i_line = 0
                commas = 0
                dtaline = ''
                dtavalid = True
            else: # Ende der Datenzeile
                if (dtavalid): # Noch die Anzahl der Komma prüfen
	                if (i_line < 27): # Die Zeile muß mit Komma enden
	                    if (commas == 16):
	                        lines[i_line] = dtaline
	                        i_line += 1
	                elif (commas == 15): # Die letzte Zeile endet nicht mit Komma
	                    lines[i_line] = dtaline
                #i_line = 0
                commas = 0
                dtaline = ''
                dtavalid = True
        elif (i_line >= 0): # Weiter im initialisierten Block
            if (char == ','):
                if (before==','):
                    dtavalid = False
                else:
                    commas += 1
            dtaline += char
        before = char
#}}} data_received()


# Innere Kommandoschleife (startet die Datenübernahme und läuft bis zu deren Ende)
# Kommandos: stop, quit (^D (EOF))
# Andere Kommandos werden ohne Fehlermeldung oder Warnung ignoriert.
# @param devinfo aus devinfos
# @return Kommando mit dem die Schleife beendet wurde.
async def innerloop(devinfo): #{{{
    is_connected = False
    command = ""

    # Callback für unerwartetes disconnect
    def disconnected(client):
        nonlocal is_connected
        if is_connected:
            print_err ("disconnected", "Unexpected disconnect")
            is_connected = False
        else:
            print_btevent ("disconnected", "Disconnected")

    async with BleakClient(devinfo['address']) as client:
        # pair/connect-notify
        if devinfo['paired']:
            print_dbg (f"{devinfo['name']} is already paired.")
        else:
            print_dbg ("Try pairing ...")
            await client.pair()
            print_dbg (f"{devinfo['name']} pair started.")
        is_connected = client.is_connected
        print_dbg (f"connected {is_connected}")
        print_dbg ("Start notify ...")
        client.set_disconnected_callback(disconnected)
        init_data ()
        await client.start_notify (UART_TX_CHAR_UUID, data_received)
        print_dbg ("Notify started.")
        print_dbg ("Enter inner commandloop ...")
        loop = asyncio.get_running_loop()
        while is_connected: #{{{ Innere Kommandoschleife. Ende mit quit oder ^D (EOF)
            print_dbg ("Inner commandloop (re)started.")
            command = await loop.run_in_executor(None, sys.stdin.readline)
            if not command:
                print_dbg ("EOF")
                command = "quit"
                is_connected = False
            command = command.rstrip ('\n')
            # Kommandos: stop quit
            if ['stop', 'quit'].count(command) > 0:
                if client.is_connected:
                    print_dbg ("Stop notify ...")
                    try:
                        await client.stop_notify(UART_TX_CHAR_UUID)
                        print_dbg ("Notify stopped.")
                    except BleakError as error:
                        print_err ("comm", f"Stop notify failed. ({error})")
                        return ""
                is_connected = False
            # Hier keine Fehlermeldung bei unbekanntem Kommando
        #}}} Innere Kommandoschleife
    # async with BleakClient(address)
    return command
#}}} innerloop()


# Die (äußere) Kommandoschleife
# initialisiert den Controllerstatus und erwartet danach Kommandos.
# Kommandos: status, scan, notify <name>, stop, quit, set <XY>
async def commandloop(): #{{{
    global opt_not_null
    global devices
    global devinfos

    print_dbg ("Enter outer commandloop ...")
    # Initialisierungen
    devices = []
    devinfos = []
    command = ""
    sys.stdin.flush()
    loop = asyncio.get_running_loop()
    while command != "quit": #{{{ Äußere Kommandoschleife
        print_dbg ("Outer commandloop (re)started.")
        line = await loop.run_in_executor(None, sys.stdin.readline)
        if not line:
            print_dbg ("EOF")
            command = "quit"
            break
        if line == "\n":
            continue
        command = line.rstrip ('\n').split()[0]
        args = line.rstrip ('\n').split()[1:]
        if command == "status":
            # Status ausgeben (devinfos)
            json.dump(devinfos, sys.stdout); print("", flush=True)
        elif command == "scan":
            # Scan ausführen, devinfos ableiten, devinfos ausgeben
            devices = await BleakScanner.discover()
            devinfos = []
            for dev in devices:
                name = dev.name
                if name == None:
                    continue
                    
                if name[0:8] != "SmartCov":
                    continue
                address = dev.address
                props = dev.details['props']
                paired = props['Paired']
                connected = props['Connected']
                devinfos.append({'name': name, 'address': address, 'paired': paired, 'connected': connected})
            json.dump(devinfos, sys.stdout); print("", flush=True)
        elif command == "notify": # name
            if (len(args) != 1):
                print_err ("client", f"Invalid args ({args}). Must be notify <controller_name>.")
                continue
            found = False
            for devinfo in devinfos:
                if devinfo['name'] == args[0]:
                    found = True
                    try:
                        command = await innerloop (devinfo)
                    except Exception as error:
                        print_dbg (f"From inner loop: {error} ({type(error)})")
                        print_err ("fail", f"{error}")
            if not found:
                print_err ("fail", f"Device {args[0]} not found. Try scan or status.")
        elif command == "stop": # Dummy für restart
            continue
        elif command == "quit":
            break
        elif command == "set":
            if (len(args) != 2):
                print_err ("client", f"Invalid args ({args}). Must be set nulls on|off.")
                continue
            if args[0] != "nonulls":
                print_err ("client", f"Invalid option ({args[0]}). Must be set nulls ... .")
                continue
            if args[1] == "on":
                opt_not_null = True
            elif args[1] == "off":
                opt_not_null = False
            else:
                print_err ("client", f"Invalid value ({args[1]}). Must be set nulls on|off.")
        elif command == "version":
            json.dump({'btevent': 'version', 'version': VERSION}, sys.stdout); print("", flush=True)
        else:
            print_err ("client", f"Unknown command ({command})")
    #}}} Äußere Kommandoschleife
    print_dbg ("Outer commandloop exited.")
#}}} commandloop()


# Die Hauptschleife
# ruft die Kommandoschleife auf und fängt dort evtl. aufgetretene unerwarte Fehler auf,
# um sie danach neu zu starten.
# Nach einem regulären Ende der Kommandoschleife wird auch die Hauptschleife beendet.
def mainloop(): #{{{
    while True:
        try:
            asyncio.run(commandloop())
            print_dbg ("commandloop() finished.")
            break
        # expect asyncio.exceptions.TimeoutError
        except Exception as error:
            print_err (f"internal", f"From commandloop(): {error} ({type(error)})")
            continue
#}}} mainloop()

mainloop()
print_dbg ("mainloop() finished.")

