# custom/kernel/db.tcl
# Kernelmodul für die lokalen Datenbanken
#
# Erforderliche DEBIAN-Pakete:
#   tcl8.6-tdbc
#   tcl8.6-tdbc-sqlite3
#
# Das aktuelle TDBC Verbindungskommando ist $::kernel::DB::clients
#

package require tdbc
package require tdbc::sqlite3
package require json

namespace eval DB {
    variable clients ""
    variable settings ""

    namespace export nextId updateNextids dictlist2json currentTimestamp currentDate 


    # Nächste Id für eine Tabelle in $clients erfragen
    proc nextId {table_name} {; #{{{
        variable clients

        if {"$clients" == ""} {
            error "Keine Datenbankverbindung"
        }
        set next_id ""
        set result [$clients allrows "SELECT next_id FROM nextids WHERE table_name='${table_name}'"]
        if {[llength $result] == 1} {
            set next_id [dict get [lindex $result 0] next_id]
        } else {
            error "Cannot determine next_id for table $table_name"
        }
        # ID erhöhen
        $clients allrows "UPDATE nextids SET next_id=next_id+1 WHERE table_name='${table_name}'"
        return $next_id
        #}}}
    }; # proc nextId 


    #TODO proc updateNextids (export)

    #TODO sessions bereinigen
    #       leere löschen
    #       initial_situation2 und final_situation2 aktualisieren, falls nötig


    #TODO proc kvlist2json (args)


    # TCL-Liste in JSON Array umwandeln
    # @param lst    TCL-Liste
    # @return       JSON-Array
    proc list2JSONArray {lst} {; #{{{
        set ja [list]
        foreach item $lst {
            lappend ja "\"[string map {\" \\\"} $item]\""
        }
        return "\[[join $ja {, }]\]"
        #}}}
    }; # proc list2JSONArray 


    # Unterprogramm zu setAppsettings
    proc dict2JSONsetting {key value} {; #{{{
        srvLog [namespace current]::dict2JSONsetting Debug "$key: $value"
        if {[llength $value] == 2} {
            if {[dict exists $value value]} {
                return "\"$key\": \"[dict get $value value]\""
            }
        }
        set jsonsettings ""
        dict for {subkey subvalue} $value {
            if {"$jsonsettings" != ""} {
                append jsonsettings ", "
            }
            append jsonsettings [dict2JSONsetting $subkey $subvalue]
        }
        return "\"$key\": {$jsonsettings}"
        #}}}
    }; # proc dict2JSONsetting 


    # Die Einstellungen einer App in vlbsettings hinterlegen.
    # Das in der Variablen "appsettings" übergebene Dictionary enthält
    # (evtl. verschachtelte) Objekte, deren innerstes den Schlüssel "value" hat.
    # Bei der Umformung in ein äquivalentes JSON Objekt wird der zu "value"
    # gehörende Wert als String direkt dem übergeordneten Schlüssel zugeordnet.
    # @param appname     Name der App
    # @param appsettings Die Einstellungen als Dictionary
    proc setAppsettings {appname appsettings} {; #{{{
        variable settings

        set jsonsettings ""
        dict for {key value} $appsettings {
            if {"$jsonsettings" != ""} {
                append jsonsettings ", "
            }
            append jsonsettings [dict2JSONsetting $key $value]
        }
        srvLog [namespace current]::setAppsettings Notice "Settings for $appname:\n{$jsonsettings}"
        if {[catch {
            $settings begintransaction
            set sql "UPDATE applications SET appsettings='{[string map {' ''} $jsonsettings]}' WHERE name = '$appname'"
            srvLog [namespace current]::setAppsettings Debug $sql
            $settings allrows $sql
            set changes [$settings allrows -as lists "SELECT changes()"]
            if {$changes == 0} {
                srvLog [namespace current]::setAppsettings Warn "App '$appname' is not initialized. No settings stored."
            }
            $settings commit
        } msg]} {
            $settings rollback
            srvLog [namespace current]::setAppsettings Error $msg
        }
        #}}}
    }; # proc setAppsettings 


    # Die in vlbsettings hinterlegten Einstellungen einer App holen
    # @param appname    Name der App
    # @return           Die Einstellungen als Dictionary.
    proc getAppsettings {appname} {; #{{{
        variable settings

        set sql "SELECT appsettings FROM applications WHERE name='$appname'"
        srvLog [namespace current]::getAppsettings Debug $sql
        set appsettings [$settings allrows -as lists $sql]
        if {[llength $appsettings] == 0} {
            srvLog [namespace current]::getAppsettings Warn "Appname '$appname' not found in table applications"
            # App anlegen
            set sql "SELECT next_id FROM nextids WHERE table_name = 'applications'"
            srvLog [namespace current] Debug $sql
            set app_ids [$settings allrows -as lists $sql]
            set app_id [lindex $app_ids 0]
            set sql "UPDATE nextids SET next_id=next_id+1 WHERE table_name = 'applications'"
            srvLog [namespace current] Debug $sql
            $settings allrows $sql
            set sql "INSERT INTO applications (app_id, name) VALUES ($app_id, '$appname')"
            srvLog [namespace current] Debug $sql
            $settings allrows $sql
            srvLog [namespace current]::getAppsettings Info "Appname '$appname' added in table applications"
            return [dict create]
        }
        return [json::json2dict [lindex [lindex $appsettings 0] 0]]
        #}}}
    }; # proc getAppsettings 


    # Nur vom App-Client benötigte Einstellungen einer App in vlbsettings hinterlegen.
    # @param appname    Name der App
    # @param jsonsettings   Die Einstellungen als JSON-String
    # @return   Leerstring oder Fehlermeldung
    proc setClientsettings {appname clientsettings} {; #{{{
        variable settings

        if {[catch {
            $settings begintransaction
            set sql "UPDATE applications SET clientsettings=json('{[string map {' ''} $clientsettings]}') WHERE name = '$appname'"
            srvLog [namespace current]::setClientsettings Debug $sql
            $settings allrows $sql
            set changes [$settings allrows -as lists "SELECT changes()"]
            if {$changes == 0} {
                srvLog [namespace current]::setClientsettings Warn "App '$appname' is not initialized. No settings stored."
            }
            $settings commit
        } msg]} {
            $settings rollback
            srvLog [namespace current]::setClientsettings Error $msg
            return "Database error s. logfile"
        }
        return ""
        #}}}
    }; # proc setClientsettings 


    # Nur vom App-Client benötigte Einstellungen einer App aus vlbsettings holen.
    # @param appname    Name der App
    # @return           Die Einstellungen als JSON-String
    proc getClientsettings {appname} {; #{{{
        variable settings

        set clientsettings "{}"
        if {[catch {
            set sql "SELECT clientsettings FROM applications WHERE name ='$appname'"
            srvLog [namespace current]::getClientsettings Debug $sql
            set clientsettings [$settings allrows -as lists $sql]
            if {[llength $clientsettings] > 0} {
                set clientsettings [lindex [lindex $clientsettings 0] 0]
            }
        } msg]} {
            srvLog [namespace current]::getClientsettings Error $msg
            #TODO Der client sollte erfahren, daß etwas schiefgegangen ist.
            #   => Leerstring bei Fehler?
        }
        return $clientsettings
        #}}}
    }; # proc getClientsettings 


    # Die evtl. von einer App aus vmkprodukte verwendeten Fragen initialisieren
    # @param appname        Name der App
    # @param appsections    Dictionary:
    #           keys    appnames
    #           values  appsections
    #               keys pos_nr, codes(dict), texte(list) 
    #               codes keys code, values pos_nr    
    proc initAppquestions {appname appsections} {; #{{{
        variable settings

        srvLog [namespace current] Info "Initializing appquestions for $appname ..."
        if {[catch {
            $settings begintransaction
            # Ist appname bereits in applications ?
            set sql "SELECT app_id FROM applications WHERE name = '$appname'"
            srvLog [namespace current] Debug $sql
            set app_ids [$settings allrows -as lists $sql]
            if {[llength $app_ids] == 0} {; # App muß noch eingetragen werden.
                set sql "SELECT next_id FROM nextids WHERE table_name = 'applications'"
                srvLog [namespace current] Debug $sql
                set app_ids [$settings allrows -as lists $sql]
                set app_id [lindex $app_ids 0]
                set sql "UPDATE nextids SET next_id=next_id+1 WHERE table_name = 'applications'"
                srvLog [namespace current] Debug $sql
                $settings allrows $sql
                set sql "INSERT INTO applications (app_id, name) VALUES ($app_id, '$appname')"
                srvLog [namespace current] Debug $sql
                $settings allrows $sql
            } else {; # App ist bereits eingetragen.
                set app_id [lindex $app_ids 0]
            }
            dict for {code appsection} $appsections {
	            # Ist code bereits in appsections ?
                set sql "SELECT appsection_id FROM appsections WHERE app_id = $app_id AND code = '$code'"
                srvLog [namespace current] Debug $sql
                set appsection_ids [$settings allrows -as lists $sql]
                if {[llength $appsection_ids] == 0} {; # appsection neu eintragen
                    set sql "SELECT next_id FROM nextids WHERE table_name = 'appsections'"
                    srvLog [namespace current] Debug $sql
                    set appsection_ids [$settings allrows -as lists $sql]
                    set appsection_id [lindex $appsection_ids 0]
                    set sql "UPDATE nextids SET next_id=next_id+1 WHERE table_name = 'appsections'"
                    srvLog [namespace current] Debug $sql
                    $settings allrows $sql
                    set sql "INSERT INTO appsections (appsection_id, app_id, pos_nr, code, texte) \                                 VALUES ($appsection_id, $app_id, [dict get $appsection pos_nr], \
                                        '$code', \
                                        '[string map {' ''} [list2JSONArray [dict get $appsection texte]]]')"
                    srvLog [namespace current] Debug $sql
                    $settings allrows $sql
                } else {; # pos_nr, texte aktualisieren
                    set appsection_id [lindex $appsection_ids 0]
                    set sql "UPDATE appsections \
                                SET pos_nr=[dict get $appsection pos_nr], \
                                    texte='[string map {' ''} [list2JSONArray [dict get $appsection texte]]]'\
                             WHERE appsection_id = $appsection_id AND app_id=$app_id"
                    srvLog [namespace current] Debug $sql
                    $settings allrows $sql
                }
	            # Andere Fragen als in der Datenbank ?
                set codes [dict get $appsection codes]
                set sql "SELECT count(*) AS count FROM appquestions aq JOIN fragen f ON f.frage_id=aq.frage_id WHERE f.code IN ('[join [dict keys $codes] {', '}]')"
                srvLog [namespace current] Debug $sql
                set count [lindex [$settings allrows -as lists $sql] 0]
                if {$count == [dict size $codes]} {; # Keine anderen Fragen
                    srvLog [namespace current] Debug "Fragen unverändert => nur pos_nr aktualisieren"
                    dict for {code pos_nr} $codes {
                        set sql "UPDATE appquestions SET pos_nr=$pos_nr WHERE appsection_id=$appsection_id AND frage_id=(SELECT frage_id FROM fragen WHERE code='$code')"
                        srvLog [namespace current] Debug $sql
                        $settings allrows $sql
                    }
                } else {; # Auswahl der Fragen geändert.
                    srvLog [namespace current] Debug "Fragen gändert => neu anlegen"
                    set sql "DELETE FROM appquestions WHERE appsection_id = $appsection_id"
                    srvLog [namespace current] Debug $sql
                    $settings allrows $sql
                    dict for {code pos_nr} $codes {
                        set sql "INSERT INTO appquestions (appsection_id, frage_id, pos_nr) \
                                    VALUES ($appsection_id, (SELECT frage_id FROM fragen WHERE code='$code'), $pos_nr)"
                        srvLog [namespace current] Debug $sql
                        $settings allrows $sql
                    }
                }
            }; # dict for ... appsections 
            $settings commit
            srvLog [namespace current] Info "appquestions initialized for $appname."
        } msg]} {
            $settings rollback
            srvLog [namespace current] Error $msg
        }
        #}}}
    }; # proc initAppquestions 


    # Daten für die Fragen zu einer Anwendung abrufen
    # Liefert ein hierarchisches Objekt mit allen frageblöcken, fragen sowie antwortvorgaben
    # @param appname    Name der App
    # @param lang       (zweistelliger) Sprachcode
    # @return JSON String
    proc getAppquestions {appname lang} {; #{{{
        variable settings
        
        srvLog [namespace current]::getAppquestions Debug "$appname $lang"
        if {[catch {
            # Arrayindex zum Sprachcode ermitteln
            set sql "WITH indices (\"index\", code) AS ( \
                        SELECT \"index\", CASE WHEN code IN ('deu', 'eng') THEN substr(code,1,2) ELSE code END AS code \
                        FROM sprachindizes) \
                     SELECT coalesce ((SELECT \"index\"-1 FROM indices WHERE code = '$lang'), 0) AS aindex"
            srvLog [namespace current]::getAppquestions Debug $sql
            set aindex [lindex [$settings allrows -as lists $sql] 0]
            srvLog [namespace current]::getAppquestions Debug "Arrayindex for lang '$lang' is $aindex"
            # Fragen als JSON holen
            # Sofern der Text mit dem angeforderten Arrayindex nicht vorhanden ist, wird 0, d.h. deutsch angenommen.
            set sql "WITH vorgaben (frage_id, vorgaben) AS ( \
                        SELECT v.frage_id, \
                              json_group_array(json_object('code',v.code,'standard',CASE WHEN v.standard=0 THEN json('false') ELSE json('true') END,'text',json_extract(v.texte,CASE WHEN json_array_length(v.texte)>1 THEN '\$\[$aindex\]' ELSE '\$\[0\]' END))) AS vorgaben \
                        FROM antwortvorgaben v GROUP BY v.frage_id ORDER BY v.frage_id, v.pos_nr), \
                     ffragen (appsection_id, fragen) AS ( \
                        SELECT q.appsection_id, json_group_array(json_object('code',f.code,'antworttyp',f.antworttyp,'text',json_extract(f.texte,CASE WHEN json_array_length(f.texte)>1 THEN '\$\[$aindex\]' ELSE '\$\[0\]' END),'vorgaben',json(v.vorgaben),'antwort','')) AS fragen \
                        FROM appquestions q \
                        JOIN appsections s ON s.appsection_id=q.appsection_id \
                        JOIN fragen f ON f.frage_id=q.frage_id \
                        LEFT JOIN vorgaben v ON v.frage_id=f.frage_id \
                        GROUP BY q.appsection_id \
                        ORDER BY s.pos_nr, q.pos_nr) \
                    SELECT json_group_array(json_object('code',s.code,'text',json_extract(s.texte,CASE WHEN json_array_length(s.texte)>1 THEN '\$\[$aindex\]' ELSE '\$\[0\]' END),'fragen',json_extract(ff.fragen,'\$'))) \
                    FROM ffragen ff \
                    JOIN appsections s ON s.appsection_id=ff.appsection_id \
                    ORDER BY s.pos_nr"
            srvLog [namespace current]::getAppquestions Debug $sql
            # Erste Spalte der ersten Zeile vom Ergebnis:
            # (Das ist gleichzeitig alles.)
            set fragen [lindex [lindex [$settings allrows -as lists $sql] 0] 0]
            srvLog [namespace current]::getAppquestions Debug "fragen:\n$fragen"
        } msg]} {
            srvLog [namespace current]::getAppquestions Error $msg
            set fragen "{}"
        }
        return $fragen
        #}}}
    }; # proc getAppquestions 


    # Aktuellen Timestamp im Datenbankformat erzeugen
    proc currentTimestamp {} {; #{{{
        return [clock format [clock seconds] -format "%Y-%m-%d %H:%M:%S"]
        #}}}
    }; # proc currentTimestamp 


    # Aktuelles Datum im Datenbankformat erzeugen
    proc currentDate {} {; #{{{
        return [clock format [clock seconds] -format "%Y-%m-%d"]
        #}}}
    }; # proc currentTimestamp 


    # Datenbankverbindung zu clients schließen
    # (sinnvoll vor backup)
    proc disconnect {} {; #{{{
        variable clients

        if {$clients == ""} {
            srvLog [namespace current] Warn "Trying to disconnect from nonexistent database connection."
            return
        }
        $clients close
        set clients ""
        srvLog [namespace current] Info "Disconnected from \"/var/local/db/vlbclients.sl3db\""
        #}}}
    }; # proc disconnect 


    # Datenbankverbindung zu clients (wieder)herstellen
    # (notwendig nach backup)
    proc reconnect {} {; #{{{
        variable clients

        if {$clients != ""} {
            disconnect
        }
        if {[catch {
            tdbc::sqlite3::connection create [namespace current]::sl3conn "/var/local/db/vlbclients.sl3db"
            set clients [namespace current]::sl3conn
            srvLog [namespace current] Notice "Connected to local database \"/var/local/db/vlbclients.sl3db\""
            } msg]} {
            srvLog [namespace current] Error "Couldn't connect to local database \"/var/local/db/vlbclients.sl3db\""
        }
        #}}}
    }; # proc reconnect 


    # Initialisierung nach Laden des Moduls
    # Herstellen der lokalen Datenbankverbindungen mit SQLite3
    proc init {} {; #{{{
        variable clients
        variable settings

        # Lokale Datenbanken einbinden
        # Die Datenbanken werden initial im Verzeichnis db/ bereitgestellt.
        if {![file exists /var/local/db]} {
            srvLog [namespace current] Info "/var/local/db doesn't exist."
            exec mkdir -p /var/local/db
            srvLog [namespace current] Notice "/var/local/db created."
        }
        if {[catch {
            if {![file exists "/var/local/db/vlbclients.sl3db"]} {
                file copy $::dir/setup/vlbclients.sl3db /var/local/db/vlbclients.sl3db
                srvLog [namespace current] Notice "vlbclients.sl3db copied to /var/local/db."
            }
            if {![file exists "/var/local/db/vlbsettings.sl3db"]} {
                file copy $::dir/setup/vlbsettings.sl3db /var/local/db/vlbsettings.sl3db
                srvLog [namespace current] Notice "vlbsettings.sl3db copied to /var/local/db."
            }
        } msg]} {
            srvLog [namespace current]::init Error $msg
            return
        }
        tdbc::sqlite3::connection create [namespace current]::clients "/var/local/db/vlbclients.sl3db"
        set clients [namespace current]::clients
        srvLog [namespace current] Notice "\"/var/local/db/vlbclients.sl3db\" connected."
        tdbc::sqlite3::connection create [namespace current]::settings "/var/local/db/vlbsettings.sl3db"
        set settings [namespace current]::settings
        srvLog [namespace current] Notice "\"/var/local/db/vlbsettings.sl3db\" connected."
        #}}}
    }; # proc init 

}; # namespace eval DB 

set mod_loaded DB

