#!/usr/bin/env tclsh
# 
# import_dbldfiles.tcl
#
# Das Skript füllt die (bereits vorhandene) SqLite3 Datenbank /var/local/db/vlbclients.sl3db
# mit den (nach Empfehlung) erfaßten Sitzungen im Verzeichnis /home/horst/saddle-dbld-files.
# Es wird nur einmal, undzwar beim Update ausgeführt.
#
# Aufruf: import_dbldfiles.tcl (ohne Argumente)
#
# Das Skript muß von root ausgeführt werden.
# Bei mehrfacher Ausführung werden die Daten ebenfalls mehrfach hinzugefügt.
#

# Historie:
# 10.02.2023 Siegmar Müller fertiggestellt

package require tdbc
package require tdbc::sqlite3
package require json

package require Druckbildanalyse
package require Veloscore


set DBLDDIR "/home/horst/saddle-dbld-files"
set DBFILE "/var/local/db/vlbclients.sl3db"

if {[exec id -u] != 0} {
    puts "Must execute as 'root'"
    exit 1
}

if {![file exists $DBFILE]} {
    puts "$DBFILE doesn't exist."
    exit 1
}

tdbc::sqlite3::connection create ::db_sl3 $DBFILE
set CONN ::db_sl3
puts "Connected to $DBFILE"

# Wenn es weitere Datenbankverbindungen gibt (vmkstationd ist gestartet),
# kann trotzdem geschrieben werden.
# Während einer Transaktion wird allerdings immer die komplette Datenbank gesperrt.


namespace eval import {
    namespace export analyseSaddlePressureImages 

    variable N_ROWS 28
    variable N_COLS 16
    variable N_VORN 14; # Anzahl der Zeilen, die nach vorn gehören. (Berechnung front_rear)
    variable conn $::CONN


    # Nächste Id für eine Tabelle erfragen
    proc nextId {table_name} {; #{{{
        variable conn

        if {"$conn" == ""} {
            error "Keine Datenbankverbindung"
        }
        set next_id ""
        set result [$conn allrows "SELECT next_id FROM nextids WHERE table_name='${table_name}'"]
        if {[llength $result] == 1} {
            set next_id [dict get [lindex $result 0] next_id]
        } else {
            error "Cannot determine next_id for table $table_name"
        }
        # ID erhöhen
        $conn allrows "UPDATE nextids SET next_id=next_id+1 WHERE table_name='${table_name}'"
        return $next_id
        #}}}
    }; # proc nextId 


    # Druckdatenanalyse für den Import einer .dbld-Datei
    proc analyseDruckdaten {imagevalues maskvalues} {; #{{{
        variable N_ROWS
        variable N_COLS
        variable N_VORN 

        set analyse_std [::druckbild::analyse std $imagevalues $N_ROWS $N_COLS]
        set veloscore [::druckbild::veloscore $imagevalues $N_ROWS $N_COLS $maskvalues]
        set result [dict create]
        dict set result veloscore [format {%.1f} $veloscore]
        # analyse_std in result einbauen, soweit relevant (dabei Punkte zu einem Objekt zusammenfassen)
        dict set result center [dict create x [format {%.1f} [dict get $analyse_std schwerpunkt_x]] y [format {%.1f} [dict get $analyse_std schwerpunkt_y]]]
        dict set result center_left [dict create x [format {%.1f} [dict get $analyse_std schwerpunkt1_x]] y [format {%.1f} [dict get $analyse_std schwerpunkt1_y]]]
        dict set result center_right [dict create x [format {%.1f} [dict get $analyse_std schwerpunkt2_x]] y [format {%.1f} [dict get $analyse_std schwerpunkt2_y]]]

        #{{{ links/rechts und vorn/hinten analysieren
        set n_vorn $N_VORN
        set n_links [expr $N_COLS / 2]
        set sum_vorn 0
        set sum_hinten 0
        set sum_links 0
        set sum_rechts 0
        set sum 0
        set avg 0
        set max 0
        set n 0; # Anzahl Werte > 0
        set i_row $N_ROWS; # Zeilenindex von vorn
        # Nach vorn/hinten summieren
        # Der Sattel ist von hinten nach vorn in $imagevalues gespeichert.
        for {set i 0} {$i < [llength $imagevalues]} {incr i} {
            if {$i % $N_COLS == 0} {
                # Eine Zeile geschafft.
                incr i_row -1
                set i_col 0
            }
            incr i_col
            if {[lindex $maskvalues $i]} {
                continue
            }
            set wert [lindex $imagevalues $i]
            if {$i_row <= $n_vorn} {
                incr sum_vorn $wert
            } else {
                incr sum_hinten $wert
                # links/rechts nur hinten
                if {$i_col <= $n_links} {
                    incr sum_links $wert
                } else {
                    incr sum_rechts $wert
                }
            }
            if {$wert > $max} {
                set max $wert
            }
            if {$wert > 0} {
                incr n
            }
        }
        set sum [expr $sum_vorn + $sum_hinten]
        if {$n > 0} {
            set avg [expr $sum / ${n}.0]
        } else {; # alles 0
            set avg 0
        }
        set sum_lr [expr $sum_links + $sum_rechts]
        #}}} links/rechts und vorn/hinten analysieren

        # Ergebnis in result speichern
        dict set result front_rear "[format "%.0f" [expr $sum_vorn * 100.0 / $sum]]:[format "%.0f" [expr $sum_hinten * 100.0 / $sum]]"
        dict set result left_right "[format "%.0f" [expr $sum_links * 100.0 / $sum_lr]]:[format "%.0f" [expr $sum_rechts * 100.0 / $sum_lr]]"
        return $result
        #}}}
    }; # proc analyseDruckdaten 


    # Druckbilddatei analysieren
    # @param vrecording Variablenname des recording dict eine Stackebene tiefer
    #                   zum Eintragen des Dateiinhalts
    # @param dbld       Dateiname der Druckbilddatei
    proc analyseDBLDFile {vrecording dbld} {; #{{{
        upvar $vrecording recording

        variable N_ROWS
        variable N_COLS

        set n_rows 0
        set n_cols 0
        set i_row -1
        set lines [list]
        set druckbild [list]
        set c " "

        set fd [open $dbld "r"]
        while {[gets $fd line] > 0} {; #{{{ weitere Zeile
            set lline [split [regsub -all {  *} [string trim $line { }] { }] { }]
            set c [string range [lindex $lline 0] 0 0]
            if {![string is integer -strict $c]} {; #{{{
                if {"$c" == "#"} {; #{{{ Kommentar
                    # Email/Sattelmarke/-typ als JSON-Kommentar
                    if {[catch {set jsondata [json::json2dict [string trim $line { #}]]} result]} {
                        # Fehlerhaften JSON-String ignorieren.
                        puts "Warning, Invalid JSON ignored: $line"
                    } else {; # Texte übernehmen
                        if {[dict exists $jsondata infos]} {
                            set infos [dict get $jsondata infos]
                        } else {
                            set infos $jsondata
                        }
                        foreach {key dbcol} {fahreremail email\
                                             satteltyp product\
                                             sattelmarke product_label\
                                             notiz notes} {
                            if {[dict exists $infos $key]} {
                                dict set recording $dbcol [dict get $infos $key]
                            }
                        }
                        set pelvisrotation ""
                        if {[dict exists $jsondata beckenrotation]} {
                            set pelvisrotation "\"pelvisrotation\": {"
                            set comma ""
                            dict for {key value} [dict get $jsondata beckenrotation] {
                                if {[llength $value] == 1} {; # Einzelner Wert
                                    append pelvisrotation "${comma}\"$key\": $value"
                                } else {; # Punkt mit Koordinaten x, y
                                    append pelvisrotation "${comma}\"$key\": {\"x\": [dict get $value x], \"y\": [dict get $value y]}"
                                }
                                set comma ", "
                            }
                            append pelvisrotation "}"
                        }
                    }
                    continue
                    #}}}
                }; # if Kommentarzeile
                if {[string is upper $c]} {
                    if {"$c" == "S"} {
                        # S => Mattentyp O.K.
                    }
                    continue
                }
                puts "Unbekanntes Dateiformat"
                close $fd
                return
                #}}}
            }; # if {Zeile beginnt nicht mit Integer}
            # Zeile beginnt mit einem Integer, falls hier angekommen
            if {$i_row < 0} {; #{{{ 1. Zeile
                set i_row [string trim [lindex $lline 0] ":"]
                # Das muß eine Zahl sein, weil alles andere schon ausgeschlossen wurde.
                set n_rows [expr $i_row + 1]
                set n_cols [expr [llength $lline] - 1]
                if {$n_rows != $N_ROWS || $n_cols != $N_COLS} {
                    puts "Unerwartetes Mattenlayout($n_rows x $n_cols statt $N_ROWS x $N_COLS)"
                    close $fd
                    return
                }
                #}}}
            }; # if {1. Zeile}
            set c [string trim [lindex $lline 0] ":"]
            if {"$i_row" != $c} {
                puts "Ungültiger Zeilenindex ($c statt $i_row)"
                close $fd
                return
            }
            set c [expr [llength $lline] - 1]
            if {$c != $n_cols} {
                puts "Unerwartete Spaltenanzahl ($c statt $n_cols)"
                close $fd
                return
            }
            lappend llines $lline
            incr i_row -1
            if {$i_row < 0} {
                break
            }
            #}}}
        }; # while weitere Zeile
        close $fd

        if {$i_row >= 0} {
            puts "Unerwartetes Dateiende ([expr $i_row + 1] Zeilen fehlen)"
            return
        }

        # Zeilen zu Druckwerten zusammenbauen, dabei Maske extrahieren und Maximum bestimmen
        set maske [list]
        set max 0
        for {set i [expr $n_rows - 1]} {$i >= 0} {incr i -1} {; #{{{
            set lline [lindex $llines $i]
            foreach druck [lrange $lline 1 end] {
                if {$druck > $max} {
                    set max $druck
                }
                # Maske aus den Daten eliminieren
                lappend druckbild [string map {-1 0} $druck]
                # Maske aus den Daten extrahieren
                lappend maske [expr $druck == -1 ? 1 : 0]
            }
            #}}}
        }; # for Zeilen durchgehen

        dict set recording pressure_values $druckbild
        dict set recording max_pressure $max
        dict set recording mask $maske

        set result_analyse [analyseDruckdaten $druckbild $maske]
        set json_result "{"
        if {"$pelvisrotation" == ""} {
            set comma ""
        } else {
            append json_result $pelvisrotation
            set comma ", "
        }
        # dict $result_analyse in JSON umbauen und hinzufügen
        # siehe Umbau pelvisrotation (suchen nach 'set pelvisrotation ""')
        dict for {key value} $result_analyse {
            if {[llength $value] == 1} {; # Einzelner Wert
                append json_result "${comma}\"$key\": \"$value\""
            } else {; # Punkt mit Koordinaten x, y
                append json_result "${comma}\"$key\": {\"x\": [dict get $value x], \"y\": [dict get $value y]}"
            }
            set comma ", "
        }
        append json_result "}"
        dict set recording results $json_result
        #}}}
    }; # proc analyseDBLDFile 


    # Zwei Recordings zum aufsteigenden Sortieren nach Timestamp vergleichen
    proc compareRecordingsByDate {recording1 recording2} {
        return [expr \"[dict get $recording1 finished]\" > \"[dict get $recording2 finished]\"]
    }; # proc compareRecordingsByDate 


    # Verzeichnis für einen Kunden analysieren
    # @param vrecording Variablenname des client dict eine Stackebene tiefer
    #                   zum Eintragen des Verzeichnisinhalts
    # @param clientDir  Dateiname des Verzeichnisses
    proc analyseDirectory {vclient clientDir} {; #{{{
        upvar $vclient client

        set recordings [list]
        foreach FILE [glob -nocomplain $clientDir/*] {
            if {"[file extension $FILE]" != ".dbld"} {
                continue
            }
            set recording [dict create analysis 2 finished {} product {} product_label {} n_pressure_rows 28 n_pressure_cols 16 pressure_values {} mask {} results {} notes {}]
            # finished aus dem Dateiname entnehmen
            set basename [lindex [file split [file rootname [file rootname $FILE]]] end]
            if {[regexp {^[^_]*_[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{2}:[0-9]{2}:[0-9]{2}$} $basename]} {
                set timestamp [regsub {_} [regsub {^[^_]*_} $basename ""] " "]
                dict set recording finished [regsub {_} [regsub {^[^_]*_} $timestamp ""] " "]
            }
            analyseDBLDFile recording $FILE
            lappend recordings $recording
        }
        # recordings in sessions gruppieren
        set recordings [lsort -command compareRecordingsByDate $recordings]
        set sessions [list]
        set session [dict create date {} recordings {}]
        if {[llength $recordings] > 0} {; # Es gab überhaupt eine Session.
            set last_finished_date "0000-00-00"
            foreach recording $recordings {
                set finished_date [string range [dict get $recording finished] 0 9]
                if {"$finished_date" > "$last_finished_date"} {; # Weitere Session
                    if {[llength [dict get $session recordings]] > 0} {
                        dict set session date $last_finished_date
                        lappend sessions $session
                        set session [dict create date {} recordings {}]
                    }
                }
                dict lappend session recordings $recording
                set last_finished_date $finished_date
            }
            if {[llength [dict get $session recordings]] > 0} {
                dict set session date $finished_date
                lappend sessions $session
            }
        }

        dict set client sessions $sessions
        #}}}
    }; # proc analyseDirectory 

    # Druckbildverzeichnis für den Import analysieren
    # => Liste mit Dictionaries der Kunden zum Importieren:
    # clients:      {client ...}
    # client:       {email {} forename {} surname {} first_contact {} last_session {} sessions {}}
    # sessions:     {session ...}
    # session:      {date {} recordings {}}
    # recordings:   {recording ...}
    # recording:    {finished {} product {} product_label {} n_pressure_rows 28 n_pressurecols 16 pressure_values {} mask {} results {} notes {}}
    #               (email, falls vorhanden)
    # @param vclients   Variablenname der client list eine Stackebene tiefer
    #                   zum Eintragen des Verzeichnisinhalts
    # @param dbldDir    Dateiname des Verzeichnisses
    proc analyseSaddlePressureImages {vclients {dbldDir /home/horst/saddle-dbld-files}} {; #{{{
        upvar $vclients clients

        foreach FILE [glob [file join $dbldDir ""]/*] {
            if {[file isdirectory $FILE]} {
                set client [dict create email {} forename {} surname {} first_contact {} last_session {} sessions {}]
                # Versuche forname, surname zu ermitteln
                set name [lindex [file split [file rootname $FILE]] end]
                if {[regexp {^[^ ]+ +[^ ]+$} $name]} {
                    dict set client forename [lindex $name 0]
                    dict set client surname [lindex $name 1]
                } else {
                    dict set client surname $name
                }
                analyseDirectory client $FILE
                if {[llength [dict get $client sessions]] > 0} {; # Es gab eine Session
                    # first_contact aus recordings.finished ermitteln
                    set session1 [lindex [dict get $client sessions] 0]
                    set recording1 [lindex [dict get $session1 recordings] 0]
                    dict set client first_contact [dict get $recording1 finished]
                    # email aus den recordings übernehmen
                    # (Es wird die zuletzt eingetragene email genommen.
                    foreach session [dict get $client sessions] {
                        foreach recording [dict get $session recordings] {
                            if {[dict exists $recording email]} {
                                dict set client email [dict get $recording email]
                            }
                        }
                    }
                    lappend clients $client
                }
            }
        }
        #}}}
    }; # proc analsyseSaddlePressureImages 


    # Aktuellen Timestamp im Datenbankformat erzeugen
    proc currentTimestamp {} {
        return [clock format [clock seconds] -format "%Y-%m-%d %H:%M:%S"]
    }


    # Aktuelles Datum im Datenbankformat erzeugen
    proc currentDate {} {
        return [clock format [clock seconds] -format "%Y-%m-%d"]
    }


    # Import ausführen
    proc execute {dbldDir} {
        variable conn 

        set clients [list]
        analyseSaddlePressureImages clients $::DBLDDIR
        puts "Analysis finished, [llength $clients] clients found"
        # $clients wie bei testAnalyse abarbeiten und dabei Datensätze einfügen
        foreach client $clients {
            puts "Client: [dict get $client forename] [dict get $client surname]"
            if {"[dict get $client first_contact]" == ""} {
                dict set client first_contact [currentTimestamp]
                puts "Warning: no date for first_contact => current date assumed"
            }
            set sql "BEGIN"
            $conn begintransaction
            if {[catch {
                set sql "nextId {}"
                set client_id [nextId clients]
                dict set client client_id $client_id
                set columns [list]
                set values [list]
                dict for {column value} $client {
                    if {"$column" == "sessions"} {; # keine Spalte
                        continue
                    }
                    if {"$value" != ""} {
                        lappend columns $column
                        if {[string is double $value]} {
                            lappend values $value
                        } else {
                            lappend values '[string map {' ''} $value]'
                        }
                    }
                }
                set sql "INSERT INTO clients ([join $columns ", "]) VALUES ([join $values ", "])"
                $conn allrows $sql
                foreach session [dict get $client sessions] {
                    if {"[dict get $session date]" == ""} {
                        dict set session date [currentDate]
                        puts "Warning: no date for session => current date assumed"
                    }
                    set session_id [nextId sessions]
                    dict set session session_id $session_id
                    dict set session client_id $client_id
                    set columns [list]
                    set values [list]
                    dict for {column value} $session {
                        if {"$column" == "recordings"} {; # keine Spalte
                            continue
                        }
                        if {"$value" != ""} {
                            lappend columns $column
                            if {[string is double $value]} {
                                lappend values $value
                            } else {
                                lappend values '$value'
                            }
                        }
                    }
                    set sql "INSERT INTO sessions ([join $columns ", "]) VALUES ([join $values ", "])"
                    $conn allrows $sql
                    set recording_id1 0; # nicht mehr benutzt
                    foreach recording [dict get $session recordings] {
                        dict unset recording email
                        dict set recording notes [string map {' ''} [dict get $recording notes]]
                        set recording_id [nextId recordings]
                        if {$recording_id1 == 0} {
                            set recording_id1 $recording_id
                        }
                        dict set recording recording_id $recording_id
                        dict set recording session_id $session_id
                        set columns [list]
                        set values [list]
                        dict for {column value} $recording {
                            if {"$value" != ""} {
                                lappend columns $column
                                if {[string is double $value]} {
                                    lappend values $value
                                } else {
                                    lappend values '$value'
                                }
                            }
                        }
                        set sql "INSERT INTO recordings ([join $columns ", "]) VALUES ([join $values ", "])"
                        $conn allrows $sql
                    } 
                }; # foreach session
                $conn commit
            } result]} {
                puts "Error: $result\nExecuting $sql"
                $conn rollback
                exit 1
            }
        }; # foreach client
    }; #proc execute 


}; # namespace import


import::execute $::DBLDDIR

$CONN close

