# custom/kernel/ttysattel.tcl
# Kernelmodul für die Datenübernahme von einer Hockermatte über USB
#
# Es kann nur eine Matte bedient werden.

namespace eval TTYHocker {
    variable MIN_VALUE 10; # Druckwert unterhalb dem alles 0 gesetzt wird
    variable fd_tty ""
    variable lines [list]
    variable i_row
    variable image_handler_procnames [list]; # Weitergabe von Druckbilddaten
    variable driver_handler_procnames [list]; # Weiterreichen von Treibermeldungen
    variable active_driver_name ""

    # Callback für anliegende Daten vom TTY-Treiber
    proc handleTTY {} {; #{{{
        variable fd_tty
        variable active_driver_name
        variable lines
        variable i_row
        variable image_handler_procnames
        variable MIN_VALUE

	    set n [gets $fd_tty line]
	    if {$n < 0} {; #{{{ Fehler bei der Datenuebernahme => Abbruch
            if {[eof $fd_tty]} {
		    	if {[catch {close $fd_tty} msg]} {
	                # Die Fehlerursache wird bei close zurückgegeben.
			        srvLog $fd_tty Error "$msg"
			    } else {; # Keine Fehlermeldung evtl. aufgrund -blocking 0 (s. proc start)
                    # z.B. Error: echo_off() fehlgeschlagen: errno=25 (Inappropriate ioctl for device)
			        srvLog $fd_tty Notice "Verbindung zum Hocker getrennt ($msg)."
	            }
                set fd_tty ""
            }
	        return
	        #}}}
	    }

        if {$n == 0} {; #{{{ Leerzeile vor neuem Block
            incr i_row
            if {$i_row == 28} {; # Erwartete Blockgröße
                set n_rows $i_row
                set n_cols [llength [lindex $lines 0]]
                # Umbau für Druckbildvektor (Druckbild drehen)
                set v_sum 0
                set druckbild [list]
                for {set i_col 0} {$i_col < $n_cols} {incr i_col} {; # von hinten nach vorn
                    for {set i_row 0} {$i_row < $n_rows} {incr i_row} {; # von links nach rechts
                        set v [lindex [lindex $lines $i_row] $i_col]
                        if {$v < $MIN_VALUE} {
                            lappend druckbild 0
                        } else {
                            lappend druckbild $v
                            incr v_sum $v
                        }
                    }
                }
                # Grafikerzeugung anschieben
                # Achtung! Durch die Drehung sind jetzt n_rows und n_cols vertauscht.
                foreach image_handler_procname $image_handler_procnames {
                    $image_handler_procname $druckbild $n_cols $n_rows $v_sum
                }
            }; # else nicht die erwartete Blockgröße => Block ignorieren
            set i_row -1
            #}}}
        } else {; #{{{ (Weitere) Zeile mit Messwerten
            incr i_row
            if {[llength $lines] >= $i_row} {
                lset lines $i_row [split $line ","]
            } else {
                lappend lines [split $line ","]
            }
            #}}}
        }; # else (Weitere) Zeile mit Messwerten

        #}}}
    }; # proc handleTTY 


    # (Weiteren) Handler für Druckbild hinzufügen
    proc addImageHandler {cb_dbld_handler} {; #{{{
        variable image_handler_procnames

        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $image_handler_procnames $cb_dbld_handler] < 0} {
            lappend image_handler_procnames $cb_dbld_handler
            srvLog "[namespace current]::addImageHandler" Debug "$cb_dbld_handler hinzugefügt"
        }
        #}}}
    }; #proc addImageHandler 


    # Handler für Druckbild entfernen
    proc removeImageHandler {cb_dbld_handler} {; #{{{
        variable image_handler_procnames

        set i [lsearch $image_handler_procnames $cb_dbld_handler]
        if {$i >=0} {
            set image_handler_procnames [lreplace $image_handler_procnames $i $i]
            srvLog [namespace current] Debug "::removeImageHandler $cb_dbld_handler entfernt"
        } else {
            srvLog [namespace current] Warn "::removeImageHandler $cb_dbld_handler nicht gefunden"
        }
        #}}}
    }; # proc removeImageHandler 


    # Aktuellen TTY Status an alle Treiber Handler verteilen
    # Das kann von einer App benutzt werden, wenn ein alternativer Übertragungskanal
    # (Bluetooth, Websocket) geschlossen wurde.
    proc distributeTTYState {} {; #{{{
        variable fd_tty
        variable active_driver_name
        variable driver_handler_procnames

        set msg [dict create wsevent ttystate]
        if {"$fd_tty" == ""} {
            dict set msg state inactive
        } else {
            dict set msg state active
            dict set msg driver $active_driver_name
        }

        foreach {driver_handler} $driver_handler_procnames {
            srvLog [namespace current] Debug "$driver_handler $msg"
            $driver_handler $msg
        }
        #}}}
    }; # proc distributeTTYState 


    # (Weiteren) Handler für Treiberänderungen hinzufügen
    # Das passiert üblicherweise beim Starten einer App.
    # Dem Handler wird eine key/value-Liste mit dem wsevent "ttychange" übergeben.
    proc addDriverHandler {cb_driver_handler} {; #{{{
        variable fd_tty
        variable active_driver_name
        variable driver_handler_procnames

        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $driver_handler_procnames $cb_driver_handler] < 0} {
            lappend driver_handler_procnames $cb_driver_handler
            srvLog "[namespace current]::addDriverHandler" Debug "$cb_driver_handler hinzugefügt"
        }
        # Aktuellen Status melden
        set msg [dict create wsevent ttystate]
        if {"$fd_tty" == ""} {
            dict set msg state inactive
        } else {
            dict set msg state active
            dict set msg driver $active_driver_name
        }
        # Wahrscheinlich wird aus $msg ein Websocket-Event,
        # dessen Websocket gerade erst aktiviert wird.
        # => Verzögerung erforderlich
        after 500 "$cb_driver_handler {$msg}"
        #}}}
    }; # proc addDriverHandler 


    # Handler für Treiberänderung entfernen
    proc removeDriverHandler {cb_driver_handler} {; #{{{
        variable driver_handler_procnames

        set i [lsearch $driver_handler_procnames $cb_driver_handler]
        if {$i >=0} {
            set driver_handler_procnames [lreplace $driver_handler_procnames $i $i]
            srvLog [namespace current] Debug "::removeDriverHandler $cb_driver_handler entfernt"
        } else {
            srvLog [namespace current] Warn "::removeDriverHandler $cb_driver_handler nicht gefunden"
        }
        #}}}
    }; # proc removeDriverHandler 


    # Start der Datenübernahme vom Hocker
    # @param driver_name     Name des Treibers in /dev
    #                        (wird als id der Matte verwendet)
    # @param speed           Baudrate als Integer
    # @param cb_dbld_handler Name der Callback Prozedur für die Übergabe eines Druckbildes
    #                        Argumente: druckbild n_rows n_cols v_sum
    # @return                1 bei Erfolg, 0 bei Fehler
    proc start {driver_name speed {cb_dbld_handler {}} } {; #{{{
        variable fd_tty
        variable i_row
        variable image_handler_procnames
        variable active_driver_name

        if {"$active_driver_name" != ""} {
            srvLog [namespace current]::start Warn "Neue Matte auf $driver_name ignoriert. Bediene bereits $active_driver_name"
            return 0
        }
        #TODO interner Fehler, wenn $fd_tty != ""
        if {[catch {
            open "|${::BINDIR}/read_tty -B $speed -n $driver_name" r
            } fd_tty]} {
            srvLog [namespace current]::start Error "$fd_tty"
            set fd_tty ""
            return 0
        }
        # -blocking 0 führt im Fehlerfall dazu, daß bei close kein Fehler ausgelöst wird, auch wenn es einen gab.
        # => Im Zweifel zum Testen auskommentieren.
        fconfigure $fd_tty -blocking 0
        fileevent $fd_tty readable [list [namespace current]::handleTTY]
        set active_driver_name $driver_name
        set i_row -1
        if {"$cb_dbld_handler" != ""} {
            lappend image_handler_procnames $cb_dbld_handler
        }
        srvLog $fd_tty Notice "Datenübernahme vom Hocker gestartet. (${::BINDIR}/read_tty, PID=[pid $fd_tty])"
        return 1
        #}}}
    }; # proc start 


    # Handler (callback) für Meldungen von watchTTYACM
    # (Gemeldet werden nur Änderungen zu Standard(Hocker)-matten.)
    # @change Dict mit der von watchTTYACM::handleWatch{} gemeldeten Änderung
    #           Schlüssel:
    #               action      +|- TODO evtl. I E
    #               driver      ttyACM<nr>
    #               type        Mattentyp
    #               speed       Serielle Geschwindigkeit
    proc driverChanged {change} {; #{{{
        variable fd_tty
        variable active_driver_name
        variable driver_handler_procnames

        srvLog "[namespace current]" Debug "driverChanged $change"
        set driver [dict get $change "driver"]
        # wsevent wird zur Erhaltung der Kompatibilität zunächst wie in V2 gesetzt.
        switch [dict get $change "action"] {
            + {
                # Meldung an die Clients
                set msg [list wsevent plugged driver $driver type [dict get $change type]]
                # Datenübernahme starten
                if {[start $driver [dict get $change "speed"]]} {
                    set active_driver_name $driver
                    #TODO setStationsstatus MATTE (s. TTYSattel)
                    #   => Bedeutung ändern auf "Druckdaten verfügbar" (egal woher)
                    lappend msg start ok
                } else {; # Fehler
                    lappend msg start error
                }
            }
            - {
                if {"$active_driver_name" != $driver} {
                    srvLog [namespace current] Notice "Zuvor nicht registrierter Treiber $driver entfernt."
                    return
                }
                if {"$fd_tty" != ""} {; # handleTTY hat Datenübernahme nicht gestoppet.
                    catch {close $fd_tty}
                    set fd_tty ""
                }
                set active_driver_name ""
                # Meldung an die Clients
                set msg [list wsevent unplugged driver $driver type [dict get $change type]]
                #TODO Bedeutung bei Sattelmatte unklar:  MATTE 0
            }
            default {
                return
            }
        }; # switch action
        #TODO Was machen wir damit:
        # action E: (Error)
        # action I: (Info) ?
        srvLog [namespace current] Debug "$msg"
        # Meldung an die WS-Clients als JSON-Objekt abschicken.
        # Das bleibt vorerst wegen der Kompatibilität.
        ::WSServer::disposeServerMessage messages text [::kvlist2json $msg]
        # Callbacks aufrufen
        # Dazu zunächst Schlüssel ein wenig umbauen
        set msg [dict merge $msg [dict create wsevent "ttychange" change [dict get $msg wsevent]]]
        foreach {driver_handler} $driver_handler_procnames {
            srvLog [namespace current] Debug "$driver_handler $msg"
            $driver_handler $msg
        }
        #}}}
    }; # proc driverChanged 


    # Initialisierung nach Laden des Moduls
    proc init {} {; #{{{
        variable MIN_VALUE

        srvLog [namespace current] Info "Initializing Module ..."
        # Anmelden bei watchTTYACM
        ::kernel::watchTTYACM::addDriverchangedHandler [namespace current]::driverChanged MT_STD*
        srvLog [namespace current] Info "Module initialized."
        #}}}
    }; # proc init 

}; # namespace eval TTYHocker 

set mod_loaded TTYHocker

