# custom/kernel/demosattel.tcl
# Kernelmodul für die Datenübernahme von einem gespeicherten, binären Datenstrom
#
# Die Daten werden zyklisch in einer Endlosschleife gelesen.
#
# Die Datenströme befinden sich in /var/local/rawstreams/.
# "livedata.raw" ist der zuletzt aufgezeichnete Datenstrom (noch) ohne Kommentar.
# Alle anderen Dateien <stream>.raw beginnen mit einer ASCII Kommentarzeile (Ende mit 0x0a),
# der die eigentlichen Binärdaten folgen.
#
# Das binäre Datenformat ist das für den Controller (VELOCONTROL2) definierte
# mit der Erweiterung "Timestamp":
# - Start:              0xffffffff
# - Mattenkennung:      A 0x1c 0x10
# - Timestamp:          T dd dd dd dd
# - Beschreibungsende:  0xffff
# - Daten:              hh ll hh ll ...
# - XOR-Check:          xx xx
# - Ende:               0xffff
#
# Hinweis:
#   Die "Mattenkennung" wird im verallgemeinerten Recorder-Format
#   als Datenblockdefinition angesehen.
# Historie:
# 08.11.2023 Siegmar Müller Begonnen
# 08.01.2024 Siegmar Müller Kommentar ergänzt
#

namespace eval DEMOSattel {
    variable MAX_COMMENTLENGTH 80
    variable N_ROWS 28
    variable N_COLS 16
    variable image_handler_procnames [list]; # Weitergabe von Druckbilddaten
    variable demo_handler_procnames [list]; # Weiterreichen von Demostarts/-stopps
    variable fd_stream ""; # Filedescriptor des aktiven Demostreams
    variable start_data 0; # Offset der Binärdaten im aktiven Demostream
    variable ts0 0; # System Timestamp (clock milliseconds) beim 1. Bild des wiedergegebenen Streams
    variable next_image [dict create n_rows 28 n_cols 16 values [lrepeat [expr {28*16}] 0]]
    variable n_cycles 0

    # Ermittle die verfügbaren Demo Streams
    # @return   Key/Value Liste mit <stream>/<Kommentar>
    proc streams {} {; #{{{
        variable MAX_COMMENTLENGTH

        set streams [list]
        foreach path [glob /var/local/rawstreams/*.raw] {
            set stream [regsub {^.*/([^.]*)\.raw$} $path {\1}]
            if {"$stream" == "livedata"} {
                continue
            }
            if {[catch {
                    set fd [open $path {RDONLY}]
                    set comment [regsub "\n.*" [read $fd $MAX_COMMENTLENGTH] {}]
                    if {![string is print $comment]} {; # Keine druckbare Zeichenfolge
                        error "$path: Invalid data format."
                    }
                    lappend streams $stream "$comment"
                    close $fd
                } msg]} {
                    srvLog [namespace current] Error "$msg"
            }
        }
        return $streams
        #}}}
    }; # proc streams 


    # Bereitstehendes Bild verteilen und nachfolgendes holen
    proc sendNextImage {} {; #{{{
        variable image_handler_procnames
        variable next_image

        foreach image_handler_procname $image_handler_procnames {
            $image_handler_procname [dict get $next_image values] [dict get $next_image n_cols] [dict get $next_image n_rows]
        }
        readNextData
        #}}}
    }; # proc sendNextImage 


    # Nächsten Datensatz lesen
    # und Verteilen mit Verzögerung gemäß Timestamp initiieren.
    # @param record1  1. Datensatz
    proc readNextData {{record1 0}} {; #{{{
        variable N_ROWS
        variable N_COLS
        variable fd_stream
        variable start_data
        variable ts0
        variable next_image
        variable n_cycles

        if {"$fd_stream" == ""} {; # Lesen wurde gestoppt.
            return
        }
        set descr [read $fd_stream 14]; # Beschreibungsblock
        if {[eof $fd_stream]} {
            # Zurück zum Anfang
            seek $fd_stream  $start_data
            incr n_cycles
            set record1 1
            set descr [read $fd_stream 14]; # Beschreibungsblock
        }
        set err_msg ""
        if {[binary scan $descr IaccaH8S start mtype n_rows n_cols tsmark ts end] == 7} {; # Startblock formal O.K.
            if {$start == -1 && "$mtype" eq "A" && $n_rows == $N_ROWS && $n_cols == $N_COLS && "$tsmark" eq "T" && $end == -1} {; # Startblock plausibel
                set data [read $fd_stream [expr 2 * $n_rows * $n_cols + 4 + 2]]; # Daten und Ende
                if {[binary scan $data "S[expr $n_rows * $n_cols]a4S" values check end] == 3} {; # Datenblock formal O.K.
                    # $check wird nicht geprüft.
                    if {$end == -1} {; # Alle Daten sind O.K.
                        # Datensatz für die Ausgabe ablegen (dict demoimage: values n_rows n_cols)
                        dict set next_image values $values
                        # n_rows und n_cols sind fest.
                        if {$ts == 0} {; # 1. Datensatz
                            set ts0 [clock milliseconds]
                            set delay 0
                        } {; # $ts != 0
                            # expr interpretiert $ts  mit den führenden Nullen als String.
                            set delay [expr {$ts0 + [scan $ts %d] - [clock milliseconds]}]
                        }
                        if {$delay > 0 && !$record1} {
                            after $delay [namespace current]::sendNextImage
                            return
                        } elseif {$delay == 0 && $record1} {
                            # Ausgabe sofort starten:
                            sendNextImage
                            return
                        } else {; # $delay <= 0  und/oder $record1 unplausibel
                            set err_msg "Timestamp not feasible: delay = $delay ms; record1 = $record1."
                        }
                    } else {
                        set err_msg "Invalid end mark for values."
                    }
                } else {; # Datenblock formal falsch
                    set err_msg "Values technically wrong."
                }
            } else {; # Werte im Startblock nicht plausibel
                set err_msg "Start of data not feasible. start: $start; mtype: $mtype; n_rows: $n_rows; n_cols: $n_cols; tsmark: $tsmark; end: $end"
            }
        } else {; # Startblock formal falsch
            set err_msg "Start of data technically wrong."
        }
        # Fehler, wenn hier angekommen
        srvLog [namespace current]::readNextData Error "Unexpected binary data in demo stream: '$err_msg'"
        stop
        #}}}
    }; # proc readNextData 


    # Starte die Wiedergabe eines Demostreams
    # Ein evtl. noch laufender Demostream wird zuvor beendet.
    # @param stream             Name
    # @param cb_dbld_handler    Imagehandler vor dem Verteilen des 1. Bildes setzen
    # @return    1 gestartet
    #           -1 stream existiert nicht oder ist fehlerhaft
    #            0 Fehler beim Öffnen
    proc start {stream {cb_dbld_handler {}} } {; #{{{
        variable demo_handler_procnames
        variable MAX_COMMENTLENGTH
        variable fd_stream
        variable start_data
        variable ts0
        variable n_cycles

        set streams [dict create {*}[streams]]
        if {![dict exists $streams $stream]} {
            return -1
        }
        if {[catch {
                stop 1
                set fd_stream [open "/var/local/rawstreams/${stream}.raw" {RDONLY BINARY}]
                # Start des 1. Datenblocks suchen
                set start [regsub {[^\n]*} [read $fd_stream $MAX_COMMENTLENGTH] {}]
                set start_data [expr $MAX_COMMENTLENGTH - [string length $start] + 1]; # '\n' ist noch in $start => +1
                srvLog [namespace current]::start Debug "start_data in ${stream}: $start_data"
                seek $fd_stream  $start_data
                if {"$cb_dbld_handler" != ""} {
                    addImageHandler $cb_dbld_handler
                } 
                # 1. Datensatz lesen und zyklisch weitere holen und ausgeben
                # Wenn der 1. Timestamp != 0 ist, ergibt das eine riesige Verzögerung für das 1. Bild.
                # => Schadensbegrenzung (Wert wird bei 1. Timestamp == 0 präzisiert.)
                set ts0 [clock milliseconds]
                set n_cycles 0
                readNextData 1
                set msg [dict create wsevent demostate stream $stream state active]
                foreach {demo_handler} $demo_handler_procnames {
                    srvLog [namespace current]::start Debug "$demo_handler $msg"
                    $demo_handler $msg
                }
            } msg]} {
                srvLog [namespace current]::start Error "$msg"
                return 0
        }
        return 1
        #}}}
    }; # proc start 


    # Wiedergabe des aktuellen Demostreams beenden
    # @param force  Keine Warnung, wenn keine Wiedergabe gestartet wurde.
    proc stop {{force 0}} {; #{{{
        variable fd_stream
        variable n_cycles
        variable demo_handler_procnames

        if {"$fd_stream" != ""} {
            if {[catch {close $fd_stream} msg]} {
                srvLog [namespace current]::stop Warn "Error '$msg' closing demo stream."
            }
            set fd_stream ""
            srvLog [namespace current]::stop Notice "Demo stream stopped. $n_cycles cycles performed."
            set msg [dict create wsevent demostate state inactive n_cycles $n_cycles]
            foreach {demo_handler} $demo_handler_procnames {
                srvLog [namespace current]::stop Debug "$demo_handler $msg"
                $demo_handler $msg
            }
        } elseif {!$force} {
            srvLog [namespace current]::stop Warn "No active demo stream."
        }
        #}}}
    }; # proc stop 


    # (Weiteren) Handler für Druckbild hinzufügen
    proc addImageHandler {cb_dbld_handler} {; #{{{
        variable image_handler_procnames

        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $image_handler_procnames $cb_dbld_handler] < 0} {
            lappend image_handler_procnames $cb_dbld_handler
            srvLog "[namespace current]::addImageHandler" Debug "$cb_dbld_handler hinzugefügt"
        }
        #}}}
    }; #proc addImageHandler 


    # Handler für Druckbild entfernen
    # @param cb_dbld_handler    Der zu entfernende Handler
    # @param force              Keine Warnung, wenn der Handler nicht aktiv war
    proc removeImageHandler {cb_dbld_handler {force 0}} {; #{{{
        variable image_handler_procnames

        set i [lsearch $image_handler_procnames $cb_dbld_handler]
        if {$i >=0} {
            set image_handler_procnames [lreplace $image_handler_procnames $i $i]
            srvLog [namespace current] Debug "::removeImageHandler $cb_dbld_handler entfernt"
        } elseif {!$force} {
            srvLog [namespace current] Warn "::removeImageHandler $cb_dbld_handler nicht gefunden"
        }
        #}}}
    }; # proc removeImageHandler 


    # (Weiteren) Handler für Demostart/-stop hinzufügen
    # Dem Handler wird eine key/value-Liste mit dem wsevent "demostate" übergeben.
    proc addDemoHandler {cb_demo_handler} {; #{{{
        variable demo_handler_procnames

        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $demo_handler_procnames $cb_demo_handler] < 0} {
            lappend demo_handler_procnames $cb_demo_handler
            srvLog "[namespace current]::addDemoHandler" Debug "$cb_demo_handler hinzugefügt"
        }
#TODO Brauchen wir das so oder ähnlich wie bei TTY ?
#        # Aktuellen Status melden
#        set msg [dict create wsevent ttystate]
#        if {"$fd_tty" == ""} {
#            dict set msg state inactive
#        } else {
#            dict set msg state active
#            dict set msg driver $active_driver_name
#        }
#        # Wahrscheinlich wird aus $msg ein Websocket-Event,
#        # dessen Websocket gerade erst aktiviert wird.
#        # => Verzögerung erforderlich
#        after 500 "$cb_driver_handler {$msg}"
        #}}}
    }; # proc addDriverHandler 


    # Handler für Demostart/-stop entfernen
    proc removeDemoHandler {cb_demo_handler} {; #{{{
        variable demo_handler_procnames

        set i [lsearch $demo_handler_procnames $cb_demo_handler]
        if {$i >=0} {
            set demo_handler_procnames [lreplace $demo_handler_procnames $i $i]
            srvLog [namespace current]::removeDemoHandler Debug "$cb_demo_handler entfernt"
        } else {
            srvLog [namespace current]::removeDemoHandler Warn "$cb_demo_handler nicht gefunden"
        }
        #}}}
    }; # proc removeDriverHandler 


    # Initialisierung nach dem Starten des Moduls
    # (Dummy, um Warnung zu vermeiden.)
    proc init {} {; #{{{
        #}}}
    }; # proc init 

}; # namespace eval DEMOSattel 

set mod_loaded DEMOSattel
 
