# Anwendung: "Dekubitus"
# Datei: custom/apps/dekubitus.tcl
#
# Evaluierungsversion
# (Version für klinische Versuche)
#
# Offene TODOs:
#   Es handelt sich um (noch?) nicht implementierte Features,
#   bei denen noch nicht klar ist, ob sie überhaupt gebraucht werden.
#
# Historie:
# 18.12.2023 Siegmar Müller Von sitzknochenabstand.tcl übernommen
# 23.12.2023 Siegmar Müller Experimentelle Auswertung begonnen
# 28.12.2023 Siegmar Müller Experimentelle Auswertung für Messe fertig
# 29.12.2023 Siegmar Müller Befehle zum Setzen von Einstellungen
# 11.01.2023 Siegmar Müller Demoversion 0.1.0 fertig
# 13.01.2023 Siegmar Müller Version 1.0.0 mit Recorder begonnen
# 03.02.2024 Siegmar Müller Bugfix: Livebild falsch bei max. Rohwert < 254
# 16.02.2024 Siegmar Müller Kommandos zum Abspielen einer Aufzeichnung fertig
# 06.03.2024 Siegmar Müller Kommandos für Notizen zu einer Aufzeichnung fertig
# 08.03.2024 Siegmar Müller Version auf 1.0.0 gesetzt
#

package require json


namespace eval dekubitus {
    variable VERSION "1.0.0"
    variable N_ROWS 28
    variable N_COLS 16
    # Die in vlbsettings:applications.clientsettings zu speichernden Einstellungen
    # Diese werden von den diesbezüglichen Kommandos modifiziert.
    # Hier sind die defaults (die ohnehin eingestellt sind, d.h. initial keine Aktion erfordern).
    # variable "appsettings" {{{
    variable appsettings [dict create \
        min_rawvalue {"value" 104 "default" 104} \
        create_jpeg  {"value" "on" "default" "on"} \
        create_norm  {"value" "off" "default" "off"} \
        contrast {"value" 0 "default" 0} \
        bgcolor {"value" "#202040" "default" "#202040"} \
        jpegquality {"value" 60 "default" 60} \
        resolution {"value" 17 "default" 17} \
        grid {"value" "no" "default" "no"} \
        frame {"value" 0 "default" 0} \
        ]
    # }}} variable "appsettings"
    # Maximum 254, weil mit normierten Werten gearbeitet wird
    # (Nicht beim Lifebild! Dort wird 0 gesetzt)
    variable jpegoptions [dict create -quality 60 -res 17 -grid no -frame 0 -maximum 254]
    ### Dekubitusauswertung
    # Maximale Ruhezeit (Zeit mit gewissem Druck) ...
    variable MAX_REST_min 1; # ... in Minuten
    variable MAX_REST_sec [expr int($MAX_REST_min * 60)]; # ... in Sekunden
    # Maximal erforderliche Entlastungzeit (Zeit ohne Druck)
    variable MAX_RELAX_min 1; # ... in Minuten
    variable MAX_RELAX_sec [expr int($MAX_RELAX_min * 60)]; # ... in Sekunden
    variable VAL_RELAX_sec [expr 254.0 / $MAX_RELAX_sec]; # Entlastung pro Sekunde
    # Auswertungsintervall in sec
    variable INTERVAL 2
    # Alarmwiederholung mit erhöhtem Alarmlevel nach ... Minuten
    variable T_INC_ALEVEL_min 0.5
    variable N_INC_ALEVEL [expr int($T_INC_ALEVEL_min * 60 / $INTERVAL)]
    # Die Auswertung läuft
    variable started 0;     # 1 ohne, 2 mit Aufzeichnung
    variable alarmlevel 0
    variable i_inc_alevel 0
    # Timestamps ([clock seconds])
    variable ts_start
    variable ts_last
    # Durchschnittswerte im Auswertungsintervall
    variable averages [lrepeat [expr $N_ROWS * $N_COLS] 0]
    variable n_averages 0
    # Summen der normierten Druckwerte seit dem Start
    variable integrated [lrepeat [expr $N_ROWS * $N_COLS] 0]


    # Normiertes Druckbild an die Clients schicken
    # @param imagetype  Bildtyp
    # @param druckbild  Druckwerte
    # @param n_rows     Anzahl Zeilen
    # @param n_cols     Anzahl Spalten
    proc normFinished {imagetype druckbild n_rows n_cols} {; #{{{
        variable INTERVAL
        variable appsettings
        variable jpegoptions
        variable started
        variable alarmlevel
        variable N_INC_ALEVEL
        variable i_inc_alevel
        variable MAX_REST_sec
        variable VAL_RELAX_sec
        variable averages
        variable n_averages
        variable integrated
        variable ts_start
        variable ts_last

        set values [join $druckbild ","]
        # $create_norm bedeutet, daß die erzeugten Normwerte weitergegeben werden (Erzeugt werden sie immer.)
        if {[dict get $appsettings create_norm value] == "on"} {
            ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"normdata\", \"imagetype\": \"$imagetype\", \"n_rows\": $n_rows, \"n_cols\": $n_cols, \"values\": \[$values\]}"
        }
        # Ein Livebild ist während der Beobachtung evtl. noch auszuwerten.
        if {!$started} {; # Überwachung läuft nicht
            return
        }
        if {$imagetype != "dkb_live"} {
            #TODO Nur das gespeicherte Rohbild an den Recorder, falls eingeschaltet.
            return
        }

        ## Bild in Durchschnitt einbeziehen
        # Aber: 0-Bilder müssen sofort bearbeitet werden, weil kein weiteres folgt.
        set zero_image 1
        incr n_averages
        foreach v $druckbild {
            if {$v > 0} {; # Kein 0-Bild
                # Gesamtes Bild in den Durchschnitt einbeziehen.
                # (Ein 0-Bild ist wegen incr n_averages auch ohne folgende Schleife enthalten.)
                for {set i 0} {$i < $n_rows * $n_cols} {incr i} {
                    set m [expr {double([lindex $averages $i]) / $n_averages}];
                    set s [expr {double([lindex $druckbild $i]) / $n_averages}]
                
                    lset averages $i [expr {double([lindex $averages $i]) - $m + $s}]
                }
                set zero_image 0
                break
            }; # if {Kein 0-Bild}
        }

        set ts_current [clock seconds]
        set t_interval [expr {$ts_current - $ts_last}]
        if {!$zero_image} {; # Kein 0-Bild
            # Ist das Auswertungsintervall beendet?
            if {$t_interval < $INTERVAL} {; # Das Auswertungsintervall läuft noch.
                #TODO Nur das gespeicherte Rohbild an den Recorder, falls eingeschaltet.
                return
            }
        }

        ### Das Auswertungsintervall ist abgelaufen oder wir haben ein 0-Bild, falls hier angekommen.
        # => Durchschnittswerte in die Integration einbeziehen
        # f_interval: Anteil des aktuellen Intervalls an der maximalen Ruhezeit (Faktor)
        set f_interval [expr {double($t_interval) / $MAX_REST_sec}]
        # v_relax: Anteil des aktuellen Intervalls an der Entlastung (Subtrahend)
        set v_relax [expr $VAL_RELAX_sec * $t_interval]
        srvLog [namespace current]::normFinished Info "Intervall ${t_interval}s = [expr {$f_interval * 100}]% vorbei."
        set max_reached 0
        for {set i 0} {$i < $n_rows * $n_cols} {incr i} {
            set v [lindex $averages $i]
            set s [lindex $integrated $i]
            if {$v == 0} {; # nichts aufzusummieren
                if {$s > 0} {; # Summe gemäß Entlastungsdauer reduzieren 
                    if {$s > $v_relax} {
                        lset integrated $i [expr $s - $v_relax]
                    } else {
                        lset integrated $i 0
                    }
                }; # else Bereits vollständig entlastet.
            } elseif {$s < 254} {
                set s [expr {$s + $f_interval * $v}]
                if {$s < 254} {
                    lset integrated $i $s
                } else {
                    lset integrated $i 254
                    set max_reached 1
                }
            } else {; # Maximum ist schon erreicht
                set max_reached 1
            }
        }
        if {$max_reached} {
            if {$alarmlevel == 0} {
                set alarmlevel 1
                set i_inc_alevel 0
            } else {
                if {[incr i_inc_alevel] >= $N_INC_ALEVEL} {
                    incr alarmlevel
                    set i_inc_alevel 0
                }
            }
            if {$i_inc_alevel == 0} {
                ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"alarm\", \"level\": $alarmlevel}"
                set result [::kernel::Recorder::writeNote "Alarm level set to $alarmlevel"]
                if {"$result" != ""} {
                    srvLog [namespace current]::normFinished Error $result
                }
            }
        } else {
            if {$alarmlevel > 0} {
                set alarmlevel 0
                ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"alarm\", \"level\": 0}"
                set result [::kernel::Recorder::writeNote "Alarm reset"]
                if {"$result" != ""} {
                    srvLog [namespace current]::normFinished Error $result
                }
            }
        }
        # Integriertes Druckbild mit Integerwerten
        set int_int [list]
        foreach sv $integrated {
            lappend int_int [expr round($sv)]
        }
        if {$started == 2} {; # Aufzeichnng eingeschaltet
            # averages runden (=> avg_int)
            set int_avg [list]
            foreach sv $averages {
                lappend int_avg [expr round($sv)]
            }   
            # avg_int mit int_int zum Recorder schicken.
            set result [::kernel::Recorder::record [list $int_avg $int_int]]
        }
        # JPEG-Generierung (falls eingeschaltet) starten
        if {[dict get $appsettings create_jpeg value] == "on"} {
            ::kernel::JPEGSattel::createJPEG dkb_int $int_int $n_rows $n_cols $jpegoptions
        }
        # Normierte Werte (falls eingeschaltet) verteilen.
        if {[dict get $appsettings create_norm value] == "on"} {
            set values [join $int_int ","]
            ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"normdata\", \"imagetype\": \"dkb_int\", \"n_rows\": $n_rows, \"n_cols\": $n_cols, \"values\": \[$values\]}"
        }
        if {$zero_image} {; # 0-Bild
            if {$started == 2} {
                ::kernel::Recorder::unload 1
            }
            set started 0
            ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"stopped\", \"reason\": \"no_pressure\"}"
        } else {
            # Ring frei zur nächsten Runde
            set ts_last $ts_current
            set averages [lrepeat [expr $n_rows * $n_cols] 0]
            set n_averages 0
        }
        #}}}
    }; # proc normFinished 


    # Neues JPEG-Bild verfügbar
    # => Clients über Websocket informieren
    proc jpegFinished {imagetype} {; #{{{
        srvLog [namespace current]::jpegFinished Debug2 "$imagetype"
        ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"jpegimage\", \"imagetype\": \"$imagetype\"}"
        #}}}
    }; # proc jpegFinished 


    # Handler (Callback) für TTYSattel und BTSattel (Livebild)
    # Nimmt die neuesten Druckdaten zwecks Weiterverwendung entgegen.
    proc handleDBLDValues {druckbild n_cols n_rows} {; #{{{
        variable jpegoptions
        variable appsettings
        variable started

        # JPEG-Generierung (falls eingeschaltet) starten
        if {[dict get $appsettings create_jpeg value] == "on"} {
            # Das für die integrierten, normierten Druckwerte gesetzte (minimale) Maximum trifft hier nicht zu.
            ::kernel::JPEGSattel::createJPEG dkb_live $druckbild $n_rows $n_cols [dict replace $jpegoptions -maximum 0]
        }
        #TODO Rohdaten für Recorderaufzeichnung festhalten
        # Druckbildnormierung starten
        # (Falls nicht eingeschaltet, wird das von normFinished berücksichtigt.)
        ::kernel::NormSattel::createNorm dkb_live $druckbild $n_rows $n_cols
        #}}}
    }; # proc handleDBLDValues 


    # Ein Datenblock vom Recorder ist angekommen
    # (Callback für Recorder)
    # @param id         id (s. Doku Recorder)
    # @param druckbild  Druckwerte
    # @param n_rows     Anzahl Zeilen
    # @param n_cols     Anzahl Spalten
    proc handleRecorderValues {id druckbild n_rows n_cols} {; #{{{
        variable jpegoptions
        variable appsettings

        if {"$id" == "!"} {; # Fehler beim Abpsielen
            ::WSServer::disposeServerMessage apps/dekubitus text [::apps::createJSONError internal recorder "$druckbild"]
            return
        }
        # JPEG-Generierung (falls eingeschaltet) starten
        if {[dict get $appsettings create_jpeg value] == "on"} {
            if {"$id" == "A"} {; # Aufgezeichnetes Livebild
                # Das für die integrierten, normierten Druckwerte gesetzte (minimale) Maximum trifft hier nicht zu.
                #TODO Das gibt es momentan nicht.
                ::kernel::JPEGSattel::createJPEG dkb_live $druckbild $n_rows $n_cols [dict replace $jpegoptions -maximum 0]
            } elseif {"$id" == "v"} {; # Normiertes Durchschnitts Livebild über $INTERVAL s
                ::kernel::JPEGSattel::createJPEG dkb_live $druckbild $n_rows $n_cols $jpegoptions
            } elseif {"$id" == "i"} {; # Normiertes integriertes Bild
                ::kernel::JPEGSattel::createJPEG dkb_int $druckbild $n_rows $n_cols $jpegoptions
            }
        }
        # $create_norm bedeutet, daß die erzeugten Normwerte weitergegeben werden
        if {[dict get $appsettings create_norm value] == "on"} {
            if {"$id" == "A"} {; # Aufgezeichnetes Livebild
                #TODO Das gibt es momentan nicht. (Werte müßten erst normiert werden.)
            } elseif {"$id" == "v"} {; # Normiertes Durchschnitts Livebild über $INTERVAL s
                set values [join $druckbild ","]
                ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"normdata\", \"imagetype\": \"dkb_live\", \"n_rows\": $n_rows, \"n_cols\": $n_cols, \"values\": \[$values\]}"
            } elseif {"$id" == "i"} {; # Normiertes integriertes Bild
                set values [join $druckbild ","]
                ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"normdata\", \"imagetype\": \"dkb_int\", \"n_rows\": $n_rows, \"n_cols\": $n_cols, \"values\": \[$values\]}"
            }
        }
        #}}}
    }; # proc handleRecorderValues 


    # Die Abspielposition einer Aufzeichnung hat sich geändert.
    # (Callback für Recorder)
    # @param position   Abspielposition in %
    # @param timestamp  Timestamp in ms
    proc handlePosChange {position timestamp} {; #{{{
        srvLog [namespace current]::handlePosChange Debug "$position %; $timestamp ms"
        # Zeit in ASCII umgerechnet hinzufügen
        ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"poschange\", \"position\": $position, \"timestamp\": \"$timestamp\", \"time\": \"[::kernel::Recorder::ts2asc $timestamp]\"}"
        #}}}
    }; # proc handlePosChange 


    # Während des Abpielens einer Aufzeichnung ist eine Anmerkung aufgetaucht
    # (Callback für Recorder)
    # @param timestamp  Timestamp in ms
    # @param note       Die Anmerkung
    proc handleNote {timestamp note} {; #{{{
        srvLog [namespace current]::handleNote Debug "$timestamp: $note"
        # (Einige) Sonderzeichen in $note schützen
        set note [string map {\" \\\" \n \\n} $note]
        ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"note\", \"timestamp\": $timestamp, \"time\": \"[::kernel::Recorder::ts2asc $timestamp]\", \"note\": \"$note\"}"
        #}}}
    }; # proc handleNote 


    # Handler (Callback) für TTY-Treiberereignisse
    # @param change Änderung als wsevent (key/value-Paare)
    proc handleUSBDriverChange {change} {; #{{{
        ::WSServer::disposeServerMessage apps/dekubitus text [::kvlist2json $change]
        #}}}
    }; # proc handleUSBDriverChange 


    ##{{{ Die Kommandos (Aufruf durch handleCommand)
    #
    # Kommandos geben einen JSON- (wsevent) oder einen Leerstring zurück.
    # Falls es sich um keinen Leerstring handelt, wird die Rückgabe an die Clients verteilt.
    # Fehlermeldungen folgen dem Format von ::apps::createJSONError.
    #


    # Numerische Konfigurationseinstellungen
    #   config interval [<value>]
    #   config t_inc_alevel [<value>]
    # @param args   Liste mit 1x was? und wert
    proc configValue {args} {; #{{{
        variable INTERVAL
        variable T_INC_ALEVEL_min
        variable N_INC_ALEVEL
        variable appsettings

        if {[llength $args] > 2} {
            return [::apps::createJSONError client config "Must be: config <what> <value>"]
        }
        if {[llength $args] == 2} {; # Neuer Wert
            set value [lindex $args 1]
            if {![string is double -strict $value]} {
                return [::apps::createJSONError client config "Must be: config <what> <numval>"]
            }
            switch [lindex $args 0] {
                "interval" {
                    if {![string is integer -strict $value]} {
                        return [::apps::createJSONError client config "Must be: config interval <integer>"]
                    }
                    if {$value < 1} {
                        return [::apps::createJSONError client config "Interval must be 1 at least."]
                    }
                    dict set appsettings interval value $value
                    set INTERVAL $value
                }
                "t_inc_alevel" {
                    if {$value <= 0} {
                        return [::apps::createJSONError client config "Time must be > 0."]
                    }
                    dict set appsettings t_inc_alevel value $value
                    set T_INC_ALEVEL_min $value
                    set N_INC_ALEVEL [expr int($T_INC_ALEVEL_min * 60 / $INTERVAL)]
                }
                default {
                    return [::apps::createJSONError client config "Must be: config {interval|t_inc_alevel} ..."]
                }
            }
        }; # if {neuer Wert}
        # Aktuellen Wert zurückschicken
        set what [lindex $args 0]
        switch "$what" {
            "interval" {
                set value $INTERVAL
            }
            "t_inc_alevel" {
                set value $T_INC_ALEVEL_min
            }
            default {
                return [::apps::createJSONError client config "Must be: config {interval|t_inc_alevel}"]
            }
        }
        ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"config\", \"what\": \"$what\", \"value\": $value}"
        return ""
        #}}}
    }; # configValue 


    # Numerische Einstellungen die Dekubitusprävention direkt betreffend
    # set decubitus max_rest [<value>]
    # set decubitus max_relax [<value>]
    # args: <what> [<value>]
    # @return   Leerstring oder Fehlertext
    proc setDecubitus {args} {; #{{{
        variable MAX_REST_min
        variable MAX_REST_sec
        variable MAX_RELAX_min
        variable MAX_RELAX_sec
        variable VAL_RELAX_sec

        if {[llength $args] > 2} {
            return "Must be: set decubitus <what> <value>"
        }
        if {[llength $args] == 2} {; # Neuer Wert
            set value [lindex $args 1]
            if {![string is double -strict $value]} {
                return "Must be: set decubitus <what> <numval>"
            }
            if {$value <= 0} {
                return "Time must be > 0."
            }
            switch [lindex $args 0] {
                "max_rest" {
                    dict set appsettings max_rest value $value
                    set MAX_REST_min $value
                    set MAX_REST_sec [expr int($MAX_REST_min * 60)]
                }
                "max_relax" {
                    dict set appsettings max_relax value $value
                    set MAX_RELAX_min $value
                    set MAX_RELAX_sec [expr int($MAX_RELAX_min * 60)]
                    set VAL_RELAX_sec [expr 254.0 / $MAX_RELAX_sec]
                }
            }
        }; # if {Neuer Wert}
        # Aktuellen Wert zurückschicken
        set what [lindex $args 0]
        switch "$what" {
            "max_rest" {
                set value $MAX_REST_min
            }
            "max_relax" {
                set value $MAX_RELAX_min
            }
            default {
                return "Must be: set decubitus {max_rest|max_relax}"
            }
        }
        ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"set_decubitus\", \"what\": \"$what\", \"value\": $value}"
        return ""
        #}}}
    }; # proc setDecubitus 


    # on/off Konfigurationseinstellungen
    #   create_jpeg on|off
    #   create_norm on|off
    #   recording on|off
    # @param args   Liste mit 1x was? und wert
    proc configSwitch {args} {; #{{{
        variable appsettings

        if {[llength $args] != 2} {
            return [::apps::createJSONError client config "Must be: switch <what> on|off"]
        }
        set value [lindex $args 1]
        if {!($value in {on off})} {
            return [::apps::createJSONError client config "Must be: switch <what> on|off"]
        }
        switch [lindex $args 0] {
            "create_jpeg" {
                dict set appsettings create_jpeg value $value
            }
            "create_norm" {
                dict set appsettings create_norm value $value
            }
            "recording" {
                dict set appsettings recording value $value
            }
            default {
                return [::apps::createJSONError client config "Must be: switch create_jpeg|create_norm on|off"]
            }
        }
        return ""
        #}}}
    }; # configSwitch 


    # Kommando "set jpeg ..."
    # Syntax: set jpeg ...
    #       ... colorcontrast 0...7
    #       ... bgcolor #%02x%02x%02x
    #       ... grid no|dotted|solid
    #       ... resolution 1...???
    #       ... frame 0...???
    #       ... quality 0...100
    # @param args   s. Syntax
    # @return    Fehlermeldung oder Leerstring
    proc setJPEG {args} {; #{{{
        variable appsettings
        variable jpegoptions

        if {[llength $args] < 2} {
            return {Must be: set jpeg <what> <value>}
        }
        srvLog [namespace current]::setJPEG Debug "$args"
        set value [lindex $args 1]
        switch [lindex $args 0] {
            "colorcontrast" {
                set colorcontrast [lindex $args 1]
                if {[regexp {^[0-7]$} $colorcontrast]} {
                    ::DBLD2IMG::setGlobalJPEG colorcontrast $colorcontrast
                    dict set appsettings contrast value $colorcontrast
                    return
                } else {
                    return "Must be: set jpeg colorcontrast 0...7"
                }
            }
            "bgcolor" {
                set color [lindex $args 1]
                if {[regexp {^#[0-9A-Fa-f]{6}$} $color]} {
                    ::DBLD2IMG::setGlobalJPEG bgcolor $color
                    dict set appsettings bgcolor value $color
                    return
                } else {
                    return "Must be: set jpeg bgcolor #%02x%02x%02x"
                }
            }
            "grid" {
                if {!($value in {no dotted solid})} {
                    return "Must be: set jpeg grid no|dotted|solid"
                }
                dict set appsettings grid value $value
                dict set jpegoptions -grid $value
                return
            }
            "resolution" {
                if {[string is integer -strict $value]} {
                    if {$value >= 1} {
                        dict set appsettings resolution value $value
                        dict set jpegoptions -res $value
                        return
                    }
                }
                return "Must be: set jpeg resolution 1 ..."
            }
            "frame" {
                if {[string is integer -strict $value]} {
                    if {$value >= 0} {
                        dict set appsettings frame value $value
                        dict set jpegoptions -frame $value
                        return
                    }
                }
                return "Must be: set jpeg frame 0 ..."
            }
            "quality" {
                if {[string is integer -strict $value]} {
                    if {0 <= $value && $value <= 100} {
                        dict set appsettings jpegquality value $value
                        dict set jpegoptions -quality $value
                        return
                    }
                }
                return "Must be: set jpeg quality 0 ... 100"
            }
            default {
                return {Must be: set jpeg colorcontrast|bgcolor|grid|resolution|frame|quality <value>}
            }
        }; # switch jpeg-Option
        #TODO Letztes Bild nochmal generieren. (Das haben wir aktuell nicht mehr.)
        #}}}
    }; # Kommando "set jpeg ..."


    # Kommando "start [recording]"
    # Startet die Beobachtung
    # @return   "" oder JSON Error
    proc cmdStart {args} {; #{{{
        variable N_ROWS
        variable N_COLS
        variable started
        variable alarmlevel
        variable ts_start
        variable ts_last
        variable averages
        variable n_averages
        variable integrated
        variable jpegoptions
        variable MAX_REST_min
        variable MAX_RELAX_min

        srvLog [namespace current]::cmdStart Debug "$args"
        if {[llength $args] > 2} {
            return {Must be: start [<recording>]}
        }
        set ts_start [clock seconds]
        set ts_last $ts_start
        set averages [lrepeat [expr $N_ROWS * $N_COLS] 0]
        set n_averages 0
        set integrated [lrepeat [expr $N_ROWS * $N_COLS] 0]
        # Egal, was der Recorder gerade tut => beenden
        ::kernel::Recorder::unload 1
        if {$started} {
            ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"stopped\", \"reason\": \"restart\"}"
            set started 0
            # Wegen ::kernel::Recorder::unload ist der Recorder bereits gestoppt.
        }
        if {[llength $args] == 1} {; # mit Aufzeichnung
            # => Aufzeichnung starten
            set recording [lindex $args 0]
            #TODO Evtl. => ? Option für die Auswahl der möglichen Blocks
            # set block_A [dict create id "A" n_rows 28 n_cols 16]; # Livebild aus Rohdaten
            set block_v [dict create id "v" n_rows 28 n_cols 16]; # Durchschnitte der normierten Livedaten
            set block_i [dict create id "i" n_rows 28 n_cols 16]; # Normierte integrierte Daten
            set header "Patient id: anonymous\nObservation started: [clock format [clock seconds]]\nMAX_REST: $MAX_REST_min min\nMAX_RELAX: $MAX_RELAX_min min\n"
            set result [::kernel::Recorder::start "$recording" [list $block_v $block_i] "dekubitus" $header]
            if {$result != ""} {
                return $result
            }
            set started 2
        } else {; # else {ohne Aufzeichnung}
            set started 1
        }
        set alarmlevel 0
        ::kernel::JPEGSattel::createJPEG dkb_int $integrated $N_ROWS $N_COLS $jpegoptions
        ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"started\"}"
        return ""
        #}}}
    }; # Kommando "start"


    # Kommando "recorder ..."
    # Syntax:
    #   recorder cat [<globpattern>]
    #   recorder load <name>
    #   recorder load <name>
    #   recorder unload
    #   recorder moveto <%position>|[[h]h:][m]m:ss
    #   recorder move <offset>
    #   recorder speed <factor>
    #   recorder play
    #   recorder loop
    #   recorder pause
    #   recorder note <note>
    #   recorder delnote [<timestamp>]
    #   recorder notemarks
    #   recorder notes
    #   recorder getnote <timestamp>
    #   recorder stop
    #   recorder delete <recording>
    # @param args   Subkommando mit zugehörigen Argumenten
    # @return   "" oder JSON Error
    proc recorder {args} {; #{{{
        set result ""; # Text, wird am Schluß zu JSON Error
        set what ""
        if {[llength $args] == 0} {
            set result "Must be: recorder <subkommand> ..."
        } else {
            set what [lindex $args 0]
            set args [lrange $args 1 end]
            if {[catch {
                switch $what {
                    "cat" {; #{{{
                        set catalog [dict create]
                        if {[llength $args] == 0} {
                            set result [::kernel::Recorder::cat catalog "dekubitus"] 
                        } else {; # mit glob pattern
                            set result [::kernel::Recorder::cat catalog "dekubitus" [lindex $args 0]] 
                        }
                        if {$result == ""} {; # Liste an clients verteilen
                            # Umbau in JSON
                            set json_catalog ""
                            dict for {name header} $catalog {
                                if {"$json_catalog" != ""} {
                                    append json_catalog ", "
                                }
                                append json_catalog "\"$name\": "
                                append json_catalog "\"[string map {\" \\\" \n \\n} $header]\""
                            }
                            ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"catalog\", \"recordings\": {$json_catalog}}"
                        }
                        #}}}
                    }
                    "load" {; #{{{
                        if {[llength $args] != 1} {
                            set result "Must be: recorder load <recording>"
                        } else {; # Befehlssyntax O.K.
                            set name [lindex $args 0]
                            set result [::kernel::Recorder::load $name recording_info "dekubitus"]
                            if {"$result" == ""} {; # Kein Fehler
                                srvLog [namespace current] Debug "recording_info: $recording_info"
                                # => Umbau in JSON
                                set header [string map {\" \\\" \n \\n} [dict get $recording_info header]]
                                set n_records [dict get $recording_info n_records]
                                set blocks ""
                                foreach block [dict get $recording_info blocks] {
                                    srvLog [namespace current] Debug "block: $recording_info"
                                    if {"$blocks" != ""} {
                                        append blocks ", "
                                    }
                                    append blocks "{"
                                    set n 0
                                    dict for {key value} $block {
                                        srvLog [namespace current] Debug "key: $key"
                                        # Nur ausgewählte Angaben weitergeben
                                        if {$key in {id n_rows n_cols}} {
                                            if {$n} {
                                                append blocks ", "
                                            }
                                            append blocks "\"$key\": "
                                            srvLog [namespace current] Debug "blocks: $blocks"
                                            if {[string is integer -strict "$value"]} {
                                                append blocks $value
                                            } else {
                                                append blocks "\"$value\""
                                            }
                                            incr n
                                        }
                                    }
                                    append blocks "}"
                                }; # foreach block
                                srvLog [namespace current] Debug "blocks: $blocks"
                                set duration [dict get $recording_info duration]
                                # Dauer zum Anzeigen aufbereiten
                                set s [expr "$duration / 1000"]
                                set m [expr "$s / 60"]
                                set s [expr "$s % 60"]
                                set h [expr "$s / 60"]
                                set m [expr "$m % 60"]
                                ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"recording_info\", \"name\": \"$name\", \"header\": \"$header\", \"blocks\": \[$blocks\], \"n_records\": $n_records, \"duration\": \"[format %02d:%02d:%02d $h $m $s]\"}"
                            }; # if {Kein Fehler}
                        }; # else Befehlssyntax O.K.
                        #}}}
                    }
                    "unload" {
                        set result [::kernel::Recorder::unload 1]
                    }
                    "moveto" {; #{{{
                        if {[llength $args] != 1} {
                            set result "Must be: recorder moveto <position>"
                        } else {; # Befehlssyntax O.K.
                            set result [::kernel::Recorder::moveto $args]
                        }
                        #}}}
                    }
                    "move" {; #{{{
                        if {[llength $args] != 1} {
                            set result "Must be: recorder move <offset>"
                        } else {; # Befehlssyntax O.K.
                            set result [::kernel::Recorder::move $args]
                        }
                        #}}}
                    }
                    "speed" {; #{{{
                        if {[llength $args] != 1} {
                            set result "Must be: recorder speed <factor>"
                        } else {; # Befehlssyntax O.K.
                            set result [::kernel::Recorder::setSpeed $args]
                        }
                        #}}}
                    }
                    "play" {
                        set result [::kernel::Recorder::play]
                    }
                    "loop" {
                        set result [::kernel::Recorder::play 1]
                    }
                    "pause" {
                        set result [::kernel::Recorder::pause]
                    }
                    "note" {
                        set result [::kernel::Recorder::writeNote $args]
                    }
                    "delnote" {; #{{{ Anmerkung löschen
                        if {[llength $args] == 0} {
                            set result [::kernel::Recorder::delNote]
                        } elseif {[llength $args] == 1} {
                            set timestamp [lindex $args 0]
                            if {![string is integer -strict $timestamp]} {
                                set result "<timestamp> must be of type integer."
                            } else {
                                set result [::kernel::Recorder::delNote $timestamp]
                            }
                        } else {
                            set result "Must be: recorder delnote \[<timestamp>\]"
                        }
                        #}}}
                    }
                    "notemarks" {; #{{{ Hole die Positionen (in %) mit Notizen
                        set markers [dict create]
                        set result [::kernel::Recorder::getNoteMarks markers]
                        if {$result == ""} {; #{{{ kein Fehler => in JSON umbauen und verteilen
                            set json_markers ""
                            dict for {position notes} $markers {
                                if {"$json_markers" != ""} {
                                    append json_markers ", "
                                }
                                append json_markers "\"$position\": {\"time\": \"[dict get $notes ascts]\""
                                
                                set timestamps ""
                                foreach timestamp [dict get $notes timestamps] {
                                    if {"$timestamps" != ""} {
                                        append timestamps ", "
                                    }
                                    append timestamps $timestamp
                                }
                                append json_markers ", \"timestamps\": \[$timestamps\]}"
                            }
                            ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"notemarks\", \"notemarks\": {$json_markers}}"
                            #}}}
                        }; # if {kein Fehler}
                        #}}}
                    }
                    "notes" {; #{{{ Hole die Timestamps aller Notizen
                        set timestamps [list]
                        set result [::kernel::Recorder::getNotesTimestamps timestamps]
                        if {"$result" == ""} {; # kein Fehler => in JSON umbauen und verteilen
                            ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"notes\", \"timestamps\": \[[join $timestamps ","]\]}"
                        }
                        #}}}
                    }
                    "getnote" {; #{{{ Hole Notiz mit übergebenen Timestamp
                        set note ""
                        if {[llength $args] != 1} {
                            set result "Must be 'recorder getnote <timestamp>'"
                        } else {
                            set timestamp [lindex $args 0]
                            if {![string is integer -strict $timestamp]} {
                                set result "<timestamp> must be of type integer."
                            } else {
                                set result [::kernel::Recorder::getNote $timestamp note]
                            }
                        }
                        if {"$result" == ""} {
                            ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"note\", \"timestamp\": $timestamp, \"time\": \"[::kernel::Recorder::ts2asc $timestamp]\", \"note\": \"$note\"}"
                        }
                        #}}}
                    }
                    "stop" {
                        set result [::kernel::Recorder::stop]
                    }
                    "delete" {; #{{{ Lösche eine Aufzeichnung
                        if {[llength $args] != 1} {
                            set result "Must be: recorder delete <recording>"
                        } else {
                            set result [::kernel::Recorder::delete dekubitus [lindex $args 0]]
                        }
                        #}}}
                    }
                    default {
                        set result "Must be: recorder <subkommand> ..."
                    }
                }
                } error_msg]} {
                set result $error_msg
            }
        }
        if {"$result" != ""} {
            set result [::apps::createJSONError client "recorder $what ..." $result]
        }
        return $result
        #}}}
    }; # proc recorder

    ##}}} Die Kommandos (Aufruf durch handleCommand)


    # Eingegangenes Kommando an die Kommandoprozedur weiterleiten
    # (Aufruf von custom/wsserver/wsdomains/apps/dekubitus.tcl)
    # Bei Fehlern geht eine Meldung an die Domäne apps/dekubitus.
    # @param command    Kommando wie vom Client eingegangen
    proc handleCommand {command} {; #{{{
        variable VERSION
        variable started

	    srvLog [namespace current] Debug "Command received: $command"
        set commandname [regsub {  *.*$} $command ""]
        set args [regsub {[a-zA-Z][a-zA-Z0-9]* *} $command ""]
        set result ""

        if {[catch {
            switch $commandname {
                "config" {
                    set result [configValue {*}$args]
                }
                "switch" {
                    set result [configSwitch {*}$args]
                }
                "set" {
                    if {[llength $args] < 1} {
                        set what ""
                        set result "set what?"
                    } else {
                        set what [lindex $args 0]
                        set args [lrange $args 1 end]
                        switch $what {
                            "decubitus" {
                                set result [setDecubitus {*}$args]
                            }
                            "jpeg" {
                                set result [setJPEG {*}$args]
                            }
                            default {
                                set result "Must be: set jpeg ..."
                            }
                        }
                    }
                    if {"$result" != ""} {
                        set result [::apps::createJSONError client "set $what" $result]
                    }
                }
                "start" {; # Beobachtung (re)start
                    set result [cmdStart {*}$args]
                }
                "stop" {; # Beobachtung beenden
                    if {$started == 2} {
                        set result [::kernel::Recorder::stop]
                        #TODO Was tun bei Fehler ?
                    }
                    set started 0
                    ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"stopped\", \"reason\": \"command\"}"
                }
                "recorder" {; # Recorderkommandos
                    set result [recorder {*}$args]
                }
                "version" {
                    # $args werden stillschweigend ignoriert.
                    set version [list wsevent "version" vmkstationd $::VERSION app $VERSION]
                    ::WSServer::disposeServerMessage apps/dekubitus text [::kvlist2json $version]
                }
                default {
                    set result [::apps::createJSONError client command "Unknown command: $commandname"]
                }
            }
        } error_msg]} {; # Fehler
            if {[info exists errorInfo]} {
                append error_msg "\n$errorInfo"
            }
            #TODO Es ist vielleicht nicht notwendig, das alles an die Clients zu verteilen,
            #     es reicht eine Logmeldung. (satteldruckanalyse dto.)
            set result [::apps::createJSONError internal vmkstationd "$error_msg"]
        }
        if {"$result" != ""} {
            srvLog [namespace current] Warn "Client error: '$result'"
            ::WSServer::disposeServerMessage apps/dekubitus text $result
        } else {
            srvLog [namespace current] Info "Command executed: '$command'"
        }
        #}}}
    }; # proc handleCommand 


    ##{{{ init{}, start{}, stop{}

    # Initialisierung unmittelbar nach dem Laden
    # (Aktuell nur Dummy)
    proc init {} {; #{{{
        variable VERSION

        srvLog [namespace current] Info "Initialize App V $VERSION"
        # Keine speziellen Biblitheken.
        #srvLog [namespace current] Info "App initialized."
        #}}}
    }; # proc init 


    # Anwendung starten mit TTY
    proc start {} {; #{{{
        variable N_ROWS
        variable N_COLS
        variable appsettings
        variable jpegoptions

        srvLog [namespace current] Info "Starting App ..."
        # Callbacks setzten
        ::kernel::TTYSattel::addDriverHandler [namespace current]::handleUSBDriverChange
        ::kernel::TTYSattel::addImageHandler [namespace current]::handleDBLDValues 
        ::kernel::JPEGSattel::addImageHandler [namespace current]::jpegFinished {dkb_live dkb_int}
        #TODO ::kernel::JPEGSattel::addOverloadHandler [namespace current]::handleOverload
        ::kernel::NormSattel::addNormHandler [namespace current]::normFinished {dkb_live dkb_int}
        #TODO Overload ? (gibt es (noch?) nicht.)
        ::kernel::Recorder::addImageHandler [namespace current]::handleRecorderValues
        ::kernel::Recorder::addPosHandler [namespace current]::handlePosChange
        ::kernel::Recorder::addNoteHandler [namespace current]::handleNote
        # Einstellungen aus der Datenbank holen
        set stored_settings [::kernel::DB::getAppsettings "dekubitus"]
        srvLog [namespace current]::init Debug "appsettings: $stored_settings"
        dict for {key value} $appsettings {
            # Nur die Einstellungen übernehmen, die es tatsächlich gibt.
            # (Jemand könnte an der Datenbank gespielt haben.)
            if {[dict exists $stored_settings $key]} {
                dict set appsettings $key value [dict get $stored_settings $key]
            } else {
                dict set appsettings $key value [dict get $appsettings $key default]
            }
        }
        #TODO  Aus appsettings übernehmen:
        # max_rest  MAX_REST_min      => set MAX_REST_sec [expr int($MAX_REST_min * 60)]
        # max_relax MAX_RELAX_min     => set MAX_RELAX_sec [expr int($MAX_RELAX_min * 60)]
        # interval  INTERVAL          => set N_INC_ALEVEL [expr int($T_INC_ALEVEL_min * 60 / $INTERVAL)]
        # t_inc_alevel  T_INC_ALEVEL_min  => set N_INC_ALEVEL [expr int($T_INC_ALEVEL_min * 60 / $INTERVAL)]

        # jpegoptions ableiten
        dict set jpegoptions -quality [dict get $appsettings jpegquality value]
        dict set jpegoptions -res [dict get $appsettings resolution value]
        dict set jpegoptions -grid [dict get $appsettings grid value]
        dict set jpegoptions -frame [dict get $appsettings frame value]
        # Hintergrundfarbe und Kontrast setzen
        ::DBLD2IMG::setGlobalJPEG bgcolor [dict get $appsettings bgcolor value] colorcontrast [dict get $appsettings contrast value]
        # Leeres Bild und Positionierung mit Verzögerung schicken
        after 1000 "
            set values \[lrepeat \[expr $N_ROWS*$N_COLS\] 0\]
            ::kernel::JPEGSattel::createJPEG dkb_live \$values $N_ROWS $N_COLS {$jpegoptions}
            ::kernel::JPEGSattel::createJPEG dkb_int \$values $N_ROWS $N_COLS {$jpegoptions}
            [namespace current]::handlePosChange 0 0
            [namespace current]::handleNote 0 \"\"
        "
        srvLog [namespace current] Info "App started."
        #}}}
    }; # proc start


    # Anwendung anhalten
    proc stop {} {; #{{{
        variable appsettings

        # Callbacks zurücknehmen
        ::kernel::Recorder::removeNoteHandler [namespace current]::handleNote
        ::kernel::Recorder::removePosHandler [namespace current]::handlePosChange
        ::kernel::Recorder::removeImageHandler [namespace current]::handleRecorderValues
        ::kernel::TTYSattel::removeImageHandler [namespace current]::handleDBLDValues 
        ::kernel::TTYSattel::removeDriverHandler [namespace current]::handleUSBDriverChange
        ::kernel::JPEGSattel::removeImageHandler [namespace current]::jpegFinished
        # Abmelden beim Recorder
        ::kernel::Recorder::unload
        # Nächste Version
        #TODO ::kernel::JPEGSattel::removeOverloadHandler [namespace current]::handleOverload
        #TODO Thread für normiertes Druckbild beenden
        # appsettings ohne defaults speichern
        set as2store $appsettings
        foreach {key} [dict keys $appsettings] {
            dict unset as2store $key default
        }
        srvLog [namespace current]::stop Debug "settings to store:\n$as2store"
        ::kernel::DB::setAppsettings "dekubitus" $as2store
        srvLog [namespace current] Info "App stopped."
        #}}}
    }; # proc stop 

    ##}}} init{}, start{}, stop{}

}; # namespace eval dekubitus 

set app_loaded dekubitus

