# Reine TCL-Implentierung zum Ansteuern des Piepsers in der Velometrik-Arbeitsstation
# Einschränkungen:
#   - LEDs können nur ein- bzw. ausgeschaltet werden.
#     Dimmen erfordert das PWM (Pulse Width Modulation) Modul von WiringPi.
#   - Der Schalter kann zu bestimmten Zeitpunkten abgefragt werden,
#     aber kein Interrupt auslösen.
#     Letzteres erfordert die Interrupt-Funktion von WiringPi.
#
# Copyright: Velometrik GmbH
#
# Author: Siegmar Müller
# Historie:
#   09.10.2019 Erstellt
#   16.10.2019 Prozeduren in namespace zusammengefaßt
#

namespace eval ::piepser {

	# Die GPIO Port Pinnummern
	variable BUZZER "2"
	variable LED1 "3"
	variable LED2 "14"
	variable LED3 "4"
	variable SWITCH "15"
	
	# Schreiben in ein Element des Pseudo-Filesystems /sys/ ...
	# @param name  Elementname
	# @param value Zu schreibender Wert
	proc writeSys {name value} {
	    set fd [open $name "w"]
	    puts $fd $value
	    catch {close $fd}
	}; # proc writeSys 
	
	# GPIO-Pin für Output konfigurieren und Wert initialisieren
	# @param pin    Pinnummer
	# @param value  Anfangswert (Vorgabe 0)
	proc gpioOutput {pin {value 0}} {
	    writeSys /sys/class/gpio/unexport "$pin"
	    writeSys /sys/class/gpio/export "$pin"
	    writeSys /sys/class/gpio/gpio${pin}/direction "out"
	    writeSys /sys/class/gpio/gpio${pin}/value "$value"
	}; # proc gpioOutput 
	
	# GPIO-Pin für Input konfigurieren
	# @param pin    Pinnummer
	proc gpioInput {pin} {
	    writeSys /sys/class/gpio/unexport "$pin"
	    writeSys /sys/class/gpio/export "$pin"
	    writeSys /sys/class/gpio/gpio${pin}/direction "in"
	}; # proc gpioInput 
	
	# Piepser-Platine initialisieren
    # @return 1 Erfolg
    #         0 Fehler (wahrscheinlich kein raspi)
	proc gpioSetup {} {
		variable BUZZER
		variable LED1
		variable LED2
		variable LED3
		variable SWITCH

        if {[catch {
		    gpioOutput $BUZZER
		    gpioOutput $LED1
		    gpioOutput $LED3
		    gpioOutput $LED2
		    gpioInput $SWITCH
        }]} {
            return 0
        }
        return 1
	}; # proc gpioSetup
	
	# Aktuellen Wert eines GPIO-Pins abfragen
	# @param pin  Pinnummer
	# @return Aktueller Wert (0 oder 1)
	proc getGpioValue {pin} {
	    set fd [open /sys/class/gpio/gpio${pin}/value  "r"]
	    set value [gets $fd]
	    catch {close $fd}
	    return $value
	}; # proc getButterValue 
	
	# Wert eines GPIO-Output-Pins setzen
	# @param pin  Pinnummer
	proc setGpioValue {pin value} {
	    set fd [open /sys/class/gpio/gpio${pin}/value  "w"]
	    puts $fd $value
	    catch {close $fd}
	}; # proc setGpioValue 
	
	# Aktuellen Wert eines GPIO-Output-Pins umschalten
	# @param pin  Pinnummer
	# @param next Liste mit den Zeiten bis zum nächsten Umschalten in ms
	#             Bei leerer Liste bleibt/wird der Wert 0.
	proc toogleGpioValue {pin {next {}}} {
	    if {[llength $next] == 0} {
	        set value 0
	    } else {
	        set value [expr ![getGpioValue $pin]]
	    }
	    setGpioValue $pin $value
	    if {[llength $next] > 0} {
		after [lindex $next 0] [namespace current]::toogleGpioValue $pin "{[lrange $next 1 end]}"
	    }
	}; # proc toogleGpioValue 
	
	# LED(s) auf angegebene Farbkombination setzen
	# @param value  Bit 2 ist LED1, Bit 1 LED2 und Bit 0 LED3
	proc setLED {value} {
	    foreach {led mask} {1 4 2 2 3 1} {
	        setGpioValue [set [namespace current]::LED$led] [expr $value & $mask]
	    }
	}; # proc setLED

}; # namespace eval piepser

### Test der Prozeduren

#::piepser::gpioSetup
#
#puts "Switch: [::piepser::getGpioValue $::piepser::SWITCH]"
#
#for {set i 0} {$i < 10} {incr i} {
#    lappend music 100 100 800 2000
#}
#::piepser::toogleGpioValue $::piepser::BUZZER $music
#
#set ledvalue 0
#proc toogleLeds {} {
#    incr ::ledvalue
#    ::piepser::setLED $::ledvalue
#    after 500 toogleLeds
#}
#toogleLeds
#
## Warten, bis alles abgespielt ist
#set summe 0
#foreach interval $music {
#    incr summe $interval
#}
#incr summe 1000
#after $summe set ende 1
#vwait ende
#
