# Anwendung: "Sitzknochenabstand"
# Datei: custom/apps/dekubitus.tcl
#
# Historie:
# 18.12.2023 Siegmar Müller Von sitzknochenabstand.tcl übernommen
# 23.12.2023 Siegmar Müller Experimentelle Auswertung begonnen
# 28.12.2023 Siegmar Müller Experimentelle Auswertung für Messe fertig
# 29.12.2023 Siegmar Müller Befehle zum Setzen von Einstellungen
# 11.01.2023 Siegmar Müller Demoversion 0.1.0 fertig
#

package require json


namespace eval dekubitus {
    variable VERSION "0.1.0"
    variable N_ROWS 28
    variable N_COLS 16
    # Die in vlbsettings:applications.clientsettings zu speichernden Einstellungen
    # Diese werden von den diesbezüglichen Kommandos modifiziert.
    # Hier sind die defaults (die ohnehin eingestellt sind, d.h. initial keine Aktion erfordern).
    # variable "appsettings" {{{
    variable appsettings [dict create \
        min_rawvalue {"value" 104 "default" 104} \
        create_jpeg  {"value" "on" "default" "on"} \
        create_norm  {"value" "off" "default" "off"} \
        contrast {"value" 0 "default" 0} \
        bgcolor {"value" "#202040" "default" "#202040"} \
        jpegquality {"value" 60 "default" 60} \
        resolution {"value" 17 "default" 17} \
        grid {"value" "no" "default" "no"} \
        frame {"value" 0 "default" 0} \
        ]
    # }}} variable "appsettings"
    # Maximum 254, weil mit normierten Werten gearbeitet wird
    variable jpegoptions [dict create -quality 60 -res 17 -grid no -frame 0 -maximum 254]
    ### Dekubitusauswertung
    # Maximale Ruhezeit (Zeit mit gewissem Druck) ...
    variable MAX_REST_min 1; # ... in Minuten
    variable MAX_REST_sec [expr int($MAX_REST_min * 60)]; # ... in Sekunden
    # Maximal erforderliche Entlastungzeit (Zeit ohne Druck)
    variable MAX_RELAX_min 1; # ... in Minuten
    variable MAX_RELAX_sec [expr int($MAX_RELAX_min * 60)]; # ... in Sekunden
    variable VAL_RELAX_sec [expr 254.0 / $MAX_RELAX_sec]; # Entlastung pro Sekunde
    # Auswertungsintervall in sec
    variable INTERVAL 2
    # Alarmwiederholung mit erhöhtem alarmlevel nach ... Minuten
    variable T_INC_ALEVEL_min 0.5
    variable N_INC_ALEVEL [expr int($T_INC_ALEVEL_min * 60 / $INTERVAL)]
    # Die Auswertung läuft
    variable started 0
    variable alarmlevel 0
    variable i_inc_alevel 0
    # Timestamps ([clock seconds])
    variable ts_start
    variable ts_last
    # Durchschnittswerte im Auswertungsintervall
    variable averages [lrepeat [expr $N_ROWS * $N_COLS] 0]
    variable n_averages 0
    # Summen der normierten Druckwerte seit dem Start
    variable integrated [lrepeat [expr $N_ROWS * $N_COLS] 0]


    # Normiertes Druckbild an die Clients schicken
    # @param imagetype  Bildtyp
    # @param druckbild  Druckwerte
    # @param n_rows     Anzahl Zeilen
    # @param n_cols     Anzahl Spalten
    proc normFinished {imagetype druckbild n_rows n_cols} {; #{{{
        variable INTERVAL
        variable appsettings
        variable jpegoptions
        variable started
        variable alarmlevel
        variable N_INC_ALEVEL
        variable i_inc_alevel
        variable MAX_REST_sec
        variable VAL_RELAX_sec
        variable averages
        variable n_averages
        variable integrated
        variable ts_start
        variable ts_last

        set values [join $druckbild ","]
        #TODO "create_norm" ist nicht mehr ganz richtig.
        if {[dict get $appsettings create_norm value] == "on"} {
            ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"normdata\", \"imagetype\": \"$imagetype\", \"n_rows\": $n_rows, \"n_cols\": $n_cols, \"values\": \[$values\]}"
        }
        # Ein Livebild ist evtl. noch auszuwerten.
        if {$imagetype != "dkb_live"} {
            return
        }
        if {!$started} {; # Überwachung läuft nicht
            return
        }

        ## Bild in Durchschnitt einbeziehen
        # Aber: 0-Bilder müssen sofort bearbeitet werden, weil kein weiteres folgt.
        set zero_image 1
        incr n_averages
        foreach v $druckbild {
            if {$v > 0} {; # Kein 0-Bild
                # Gesamtes Bild in den Durchschnitt einbeziehen.
                # (Ein 0-Bild ist wegen incr n_averages auch ohne folgende Schleife enthalten.)
                for {set i 0} {$i < $n_rows * $n_cols} {incr i} {
                    set m [expr {double([lindex $averages $i]) / $n_averages}];
                    set s [expr {double([lindex $druckbild $i]) / $n_averages}]
                
                    lset averages $i [expr {double([lindex $averages $i]) - $m + $s}]
                }
                set zero_image 0
                break
            }; # if {Kein 0-Bild}
        }

        set ts_current [clock seconds]
        set t_interval [expr {$ts_current - $ts_last}]
        if {!$zero_image} {; # Kein 0-Bild
            # Ist das Auswertungsintervall beendet?
            if {$t_interval < $INTERVAL} {; # Das Auswertungsintervall läuft noch.
                return
            }
        }

        ### Das Auswertungsintervall ist abgelaufen oder wir haben ein 0-Bild, falls hier angekommen.
        # => Durchschnittswerte in die Integration einbeziehen
        # f_interval: Anteil des aktuellen Intervalls an der maximalen Ruhezeit (Faktor)
        set f_interval [expr {double($t_interval) / $MAX_REST_sec}]
        # v_relax: Anteil des aktuellen Intervalls an der Entlastung (Subtrahend)
        set v_relax [expr $VAL_RELAX_sec * $t_interval]
        srvLog [namespace current]::normFinished Info "Intervall ${t_interval}s = [expr {$f_interval * 100}]% vorbei."
        set max_reached 0
        for {set i 0} {$i < $n_rows * $n_cols} {incr i} {
            set v [lindex $averages $i]
            set s [lindex $integrated $i]
            if {$v == 0} {; # nichts aufzusummieren
                if {$s > 0} {; # Summe gemäß Entlastungsdauer reduzieren 
                    if {$s > $v_relax} {
                        lset integrated $i [expr $s - $v_relax]
                    } else {
                        lset integrated $i 0
                    }
                }; # else Bereits vollständig entlastet.
            } elseif {$s < 254} {
                set s [expr {$s + $f_interval * $v}]
                if {$s < 254} {
                    lset integrated $i $s
                } else {
                    lset integrated $i 254
                    set max_reached 1
                }
            } else {; # Maximum ist schon erreicht
                set max_reached 1
            }
        }
        if {$max_reached} {
            if {$alarmlevel == 0} {
                set alarmlevel 1
                set i_inc_alevel 0
            } else {
                if {[incr i_inc_alevel] >= $N_INC_ALEVEL} {
                    incr alarmlevel
                    set i_inc_alevel 0
                }
            }
            if {$i_inc_alevel == 0} {
                ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"alarm\", \"level\": $alarmlevel}"
            }
        } else {
            if {$alarmlevel > 0} {
                set alarmlevel 0
                ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"alarm\", \"level\": 0}"
            }
        }
        # Integriertes Druckbild mit Integerwerten
        set int_int [list]
        foreach sv $integrated {
            lappend int_int [expr round($sv)]
        }
        # JPEG-Generierung (falls eingeschaltet) starten
        if {[dict get $appsettings create_jpeg value] == "on"} {
            ::kernel::JPEGSattel::createJPEG dkb_int $int_int $n_rows $n_cols $jpegoptions
        }
        # Normierte Werte (falls eingeschaltet) verteilen.
        if {[dict get $appsettings create_norm value] == "on"} {
            set values [join $int_int ","]
            ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"normdata\", \"imagetype\": \"dkb_int\", \"n_rows\": $n_rows, \"n_cols\": $n_cols, \"values\": \[$values\]}"
        }
        if {$zero_image} {; # 0-Bild
            set started 0
            ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"stopped\", \"reason\": \"no_pressure\"}"
        } else {
            # Ring frei zur nächsten Runde
            set ts_last $ts_current
            set averages [lrepeat [expr $n_rows * $n_cols] 0]
            set n_averages 0
        }
        #}}}
    }; # proc normFinished 


    # Neues JPEG-Bild verfügbar
    # => Clients über Websocket informieren
    proc jpegFinished {imagetype} {; #{{{
        srvLog [namespace current]::jpegFinished Debug "$imagetype"
        ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"jpegimage\", \"imagetype\": \"$imagetype\"}"
        #}}}
    }; # proc jpegFinished 


    # Handler (Callback) für TTYSattel und BTSattel
    # Nimmt die neuesten Druckdaten zwecks Weiterverwendung entgegen.
    proc handleDBLDValues {druckbild n_cols n_rows} {; #{{{
        variable jpegoptions
        variable appsettings
        variable started

        # JPEG-Generierung (falls eingeschaltet) starten
        if {[dict get $appsettings create_jpeg value] == "on"} {
            ::kernel::JPEGSattel::createJPEG dkb_live $druckbild $n_rows $n_cols $jpegoptions
        }
        # Druckbildnormierung starten
        # (Falls nicht eingeschaltet, wird das von normFinished berücksichtigt.)
        ::kernel::NormSattel::createNorm dkb_live $druckbild $n_rows $n_cols
        #}}}

    }; # proc handleDBLDValues 


    # Handler (Callback) für TTY-Treiberereignisse
    # @param change Änderung als wsevent (key/value-Paare)
    proc handleUSBDriverChange {change} {; #{{{
        ::WSServer::disposeServerMessage apps/dekubitus text [::kvlist2json $change]
        #}}}
    }; # proc handleUSBDriverChange 


    ##{{{ Die Kommandos (Aufruf durch handleCommand)
    #
    # Kommandos geben einen JSON- (wsevent) oder einen Leerstring zurück.
    # Falls es sich um keinen Leerstring handelt, wird die Rückgabe an die Clients verteilt.
    # Fehlermeldungen folgen dem Format von ::apps::createJSONError.
    #


    # Numerische Konfigurationseinstellungen
    #   config interval [<value>]
    #   config t_inc_alevel [<value>]
    # @param args   Liste mit 1x was? und wert
    proc configValue {args} {; #{{{
        variable INTERVAL
        variable T_INC_ALEVEL_min
        variable N_INC_ALEVEL
        variable appsettings

        if {[llength $args] > 2} {
            return [::apps::createJSONError client config "Must be: config <what> <value>"]
        }
        if {[llength $args] == 2} {; # Neuer Wert
            set value [lindex $args 1]
            if {![string is double -strict $value]} {
                return [::apps::createJSONError client config "Must be: config <what> <numval>"]
            }
            switch [lindex $args 0] {
                "interval" {
                    if {![string is integer -strict $value]} {
                        return [::apps::createJSONError client config "Must be: config interval <integer>"]
                    }
                    if {$value < 1} {
                        return [::apps::createJSONError client config "Interval must be 1 at least."]
                    }
                    dict set appsettings interval value $value
                    set INTERVAL $value
                }
                "t_inc_alevel" {
                    if {$value <= 0} {
                        return [::apps::createJSONError client config "Time must be > 0."]
                    }
                    dict set appsettings t_inc_alevel value $value
                    set T_INC_ALEVEL_min $value
                    set N_INC_ALEVEL [expr int($T_INC_ALEVEL_min * 60 / $INTERVAL)]
                }
                default {
                    return [::apps::createJSONError client config "Must be: config {interval|t_inc_alevel} ..."]
                }
            }
        }; # if {neuer Wert}
        # Aktuellen Wert zurückschicken
        set what [lindex $args 0]
        switch "$what" {
            "interval" {
                set value $INTERVAL
            }
            "t_inc_alevel" {
                set value $T_INC_ALEVEL_min
            }
            default {
                return [::apps::createJSONError client config "Must be: config {interval|t_inc_alevel}"]
            }
        }
        ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"config\", \"what\": \"$what\", \"value\": $value}"
        return ""
        #}}}
    }; # configValue 


    # Numerische Einstellungen die Dekubitusprävention direkt betreffend
    # set decubitus max_rest [<value>]
    # set decubitus max_relax [<value>]
    # args: <what> [<value>]
    # @return   Leerstring oder Fehlertext
    proc setDecubitus {args} {; #{{{
        variable MAX_REST_min
        variable MAX_REST_sec
        variable MAX_RELAX_min
        variable MAX_RELAX_sec
        variable VAL_RELAX_sec

        if {[llength $args] > 2} {
            return "Must be: set decubitus <what> <value>"
        }
        if {[llength $args] == 2} {; # Neuer Wert
            set value [lindex $args 1]
            if {![string is double -strict $value]} {
                return "Must be: set decubitus <what> <numval>"
            }
            if {$value <= 0} {
                return "Time must be > 0."
            }
            switch [lindex $args 0] {
                "max_rest" {
                    dict set appsettings max_rest value $value
                    set MAX_REST_min $value
                    set MAX_REST_sec [expr int($MAX_REST_min * 60)]
                }
                "max_relax" {
                    dict set appsettings max_relax value $value
                    set MAX_RELAX_min $value
                    set MAX_RELAX_sec [expr int($MAX_RELAX_min * 60)]
                    set VAL_RELAX_sec [expr 254.0 / $MAX_RELAX_sec]
                }
            }
        }; # if {Neuer Wert}
        # Aktuellen Wert zurückschicken
        set what [lindex $args 0]
        switch "$what" {
            "max_rest" {
                set value $MAX_REST_min
            }
            "max_relax" {
                set value $MAX_RELAX_min
            }
            default {
                return "Must be: set decubitus {max_rest|max_relax}"
            }
        }
        ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"set_decubitus\", \"what\": \"$what\", \"value\": $value}"
        return ""
        #}}}
    }; # proc setDecubitus 


    # on/off Konfigurationseinstellungen
    # @param args   Liste mit 1x was? und wert
    proc configSwitch {args} {; #{{{
        variable appsettings

        if {[llength $args] != 2} {
            return [::apps::createJSONError client config "Must be: switch <what> on|off"]
        }
        set value [lindex $args 1]
        if {!($value in {on off})} {
            return [::apps::createJSONError client config "Must be: switch <what> on|off"]
        }
        switch [lindex $args 0] {
            "create_jpeg" {
                dict set appsettings create_jpeg value $value
            }
            "create_norm" {
                dict set appsettings create_norm value $value
            }
            default {
                return [::apps::createJSONError client config "Must be: switch create_jpeg|create_norm on|off"]
            }
        }
        return ""
        #}}}
    }; # configSwitch 


    # Kommando "set jpeg ..."
    # Syntax: set jpeg ...
    #       ... colorcontrast 0...7
    #       ... bgcolor #%02x%02x%02x
    #       ... grid no|dotted|solid
    #       ... resolution 1...???
    #       ... frame 0...???
    #       ... quality 0...100
    # @param args   s. Syntax
    # @return    Fehlermeldung oder Leerstring
    proc setJPEG {args} {; #{{{
        variable appsettings
        variable jpegoptions

        if {[llength $args] < 2} {
            return {Must be: set jpeg <what> <value>}
        }
        srvLog [namespace current]::setJPEG Debug "$args"
        set value [lindex $args 1]
        switch [lindex $args 0] {
            "colorcontrast" {
                set colorcontrast [lindex $args 1]
                if {[regexp {^[0-7]$} $colorcontrast]} {
                    ::DBLD2IMG::setGlobalJPEG colorcontrast $colorcontrast
                    dict set appsettings contrast value $colorcontrast
                    return
                } else {
                    return "Must be: set jpeg colorcontrast 0...7"
                }
            }
            "bgcolor" {
                set color [lindex $args 1]
                if {[regexp {^#[0-9A-Fa-f]{6}$} $color]} {
                    ::DBLD2IMG::setGlobalJPEG bgcolor $color
                    dict set appsettings bgcolor value $color
                    return
                } else {
                    return "Must be: set jpeg bgcolor #%02x%02x%02x"
                }
            }
            "grid" {
                if {!($value in {no dotted solid})} {
                    return "Must be: set jpeg grid no|dotted|solid"
                }
                dict set appsettings grid value $value
                dict set jpegoptions -grid $value
                return
            }
            "resolution" {
                if {[string is integer -strict $value]} {
                    if {$value >= 1} {
                        dict set appsettings resolution value $value
                        dict set jpegoptions -res $value
                        return
                    }
                }
                return "Must be: set jpeg resolution 1 ..."
            }
            "frame" {
                if {[string is integer -strict $value]} {
                    if {$value >= 0} {
                        dict set appsettings frame value $value
                        dict set jpegoptions -frame $value
                        return
                    }
                }
                return "Must be: set jpeg frame 0 ..."
            }
            "quality" {
                if {[string is integer -strict $value]} {
                    if {0 <= $value && $value <= 100} {
                        dict set appsettings jpegquality value $value
                        dict set jpegoptions -quality $value
                        return
                    }
                }
                return "Must be: set jpeg quality 0 ... 100"
            }
            default {
                return {Must be: set jpeg colorcontrast|bgcolor|grid|resolution|frame|quality <value>}
            }
        }; # switch jpeg-Option
        #TODO Letztes Bild nochmal generieren. (Das haben wir aktuell nicht mehr.)
        #}}}
    }; # Kommando "set jpeg ..."


    # Kommando "start"
    # Startet die Beobachtung
    # @return    Fehlermeldung oder Leerstring
    proc cmdStart {args} {; #{{{
        variable N_ROWS
        variable N_COLS
        variable started
        variable alarmlevel
        variable ts_start
        variable ts_last
        variable averages
        variable n_averages
        variable integrated
        variable jpegoptions

        set ts_start [clock seconds]
        set ts_last $ts_start
        set averages [lrepeat [expr $N_ROWS * $N_COLS] 0]
        set n_averages 0
        set integrated [lrepeat [expr $N_ROWS * $N_COLS] 0]
        if {$started} {
            ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"stopped\", \"reason\": \"restart\"}"
        }
        set alarmlevel 0
        set started 1
        ::kernel::JPEGSattel::createJPEG dkb_int $integrated $N_ROWS $N_COLS $jpegoptions
        ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"started\"}"
        return ""
        #}}}
    }; # Kommando "start"

    ##}}} Die Kommandos (Aufruf durch handleCommand)


    # Eingegangenes Kommando an die Kommandoprozedur weiterleiten
    # (Aufruf von custom/wsserver/wsdomains/apps/dekubitus.tcl)
    # Bei Fehlern geht eine Meldung an die Domäne apps/dekubitus.
    # @param command    Kommando wie vom Client eingegangen
    proc handleCommand {command} {; #{{{
        variable VERSION
        variable started

	    srvLog [namespace current] Debug "Command received: $command"
        set commandname [regsub {  *.*$} $command ""]
        set args [regsub {[a-zA-Z][a-zA-Z0-9]* *} $command ""]
        set result ""

        if {[catch {
            switch $commandname {
                "config" {
                    set result [configValue {*}$args]
                }
                "switch" {
                    set result [configSwitch {*}$args]
                }
                "set" {
                    if {[llength $args] < 1} {
                        set what ""
                        set result "set what?"
                    } else {
                        set what [lindex $args 0]
                        set args [lrange $args 1 end]
                        switch $what {
                            "decubitus" {
                                set result [setDecubitus {*}$args]
                            }
                            "jpeg" {
                                set result [setJPEG {*}$args]
                            }
                            default {
                                set result "Must be: set jpeg ..."
                            }
                        }
                    }
                    if {"$result" != ""} {
                        set result [::apps::createJSONError client "set $what" $result]
                    }
                }
                "start" {; # Beobachtung (re)start
                    set result [cmdStart]
                }
                "stop" {; # Beobachtung beenden
                    set started 0
                    ::WSServer::disposeServerMessage apps/dekubitus text "{\"wsevent\": \"stopped\", \"reason\": \"command\"}"
                }
                "version" {
                    # $args werden stillschweigend ignoriert.
                    set version [list wsevent "version" vmkstationd $::VERSION app $VERSION]
                    ::WSServer::disposeServerMessage apps/dekubitus text [::kvlist2json $version]
                }
                default {
                    set result [::apps::createJSONError client command "Unknown command: $commandname"]
                }
            }
        } error_msg]} {; # Fehler
            if {[info exists errorInfo]} {
                append error_msg "\n$errorInfo"
            }
            #TODO Es ist vielleicht nicht notwendig, das alles an die Clients zu verteilen,
            #     es reicht eine Logmeldung. (satteldruckanalyse dto.)
            set result [::apps::createJSONError internal vmkstationd "$error_msg"]
        }
        if {"$result" != ""} {
            srvLog [namespace current] Warn "Client error: '$result'"
            ::WSServer::disposeServerMessage apps/dekubitus text $result
        } else {
            srvLog [namespace current] Info "Command executed: '$command'"
        }
        #}}}
    }; # proc handleCommand 


    ##{{{ init{}, start{}, stop{}

    # Initialisierung unmittelbar nach dem Laden
    # (Aktuell nur Dummy)
    proc init {} {; #{{{
        variable VERSION

        srvLog [namespace current] Info "Initialize App V $VERSION"
        # Keine speziellen Biblitheken.
        #srvLog [namespace current] Info "App initialized."
        #}}}
    }; # proc init 


    # Anwendung starten mit TTY
    proc start {} {; #{{{
        variable N_ROWS
        variable N_COLS
        variable appsettings
        variable jpegoptions

        srvLog [namespace current] Info "Starting App ..."
        # Callbacks setzten
        ::kernel::TTYSattel::addDriverHandler [namespace current]::handleUSBDriverChange
        ::kernel::TTYSattel::addImageHandler [namespace current]::handleDBLDValues 
        ::kernel::JPEGSattel::addImageHandler [namespace current]::jpegFinished {dkb_live dkb_int}
        #TODO ::kernel::JPEGSattel::addOverloadHandler [namespace current]::handleOverload
        ::kernel::NormSattel::addNormHandler [namespace current]::normFinished {dkb_live dkb_int}
        #TODO Overload ? (gibt es (noch?) nicht.)
        # Einstellungen aus der Datenbank holen
        set stored_settings [::kernel::DB::getAppsettings "dekubitus"]
        srvLog [namespace current]::init Debug "appsettings: $stored_settings"
        dict for {key value} $appsettings {
            # Nur die Einstellungen übernehmen, die es tatsächlich gibt.
            # (Jemand könnte an der Datenbank gespielt haben.)
            if {[dict exists $stored_settings $key]} {
                dict set appsettings $key value [dict get $stored_settings $key]
            } else {
                dict set appsettings $key value [dict get $appsettings $key default]
            }
        }
        #TODO  Aus appsettings übernehmen:
        # max_rest  MAX_REST_min      => set MAX_REST_sec [expr int($MAX_REST_min * 60)]
        # max_relax MAX_RELAX_min     => set MAX_RELAX_sec [expr int($MAX_RELAX_min * 60)]
        # interval  INTERVAL          => set N_INC_ALEVEL [expr int($T_INC_ALEVEL_min * 60 / $INTERVAL)]
        # t_inc_alevel  T_INC_ALEVEL_min  => set N_INC_ALEVEL [expr int($T_INC_ALEVEL_min * 60 / $INTERVAL)]

        # jpegoptions ableiten
        dict set jpegoptions -quality [dict get $appsettings jpegquality value]
        dict set jpegoptions -res [dict get $appsettings resolution value]
        dict set jpegoptions -grid [dict get $appsettings grid value]
        dict set jpegoptions -frame [dict get $appsettings frame value]
        # Hintergrundfarbe und Kontrast setzen
        ::DBLD2IMG::setGlobalJPEG bgcolor [dict get $appsettings bgcolor value] colorcontrast [dict get $appsettings contrast value]
        # Leeres Bild mit Verzögerung schicken
        #TODO createJPEG und createNorm nur, wenn das in den appsettings so eingestellt ist.
        after 1000 "
            set values \[lrepeat \[expr $N_ROWS*$N_COLS\] 0\]
            ::kernel::JPEGSattel::createJPEG dkb_live \$values $N_ROWS $N_COLS {$jpegoptions}
            ::kernel::NormSattel::createNorm dkb_live \$values $N_ROWS $N_COLS
        "
        srvLog [namespace current] Info "App started."
        #}}}
    }; # proc start


    # Anwendung anhalten
    proc stop {} {; #{{{
        variable appsettings

        # Callbacks zurücknehmen
        ::kernel::TTYSattel::removeImageHandler [namespace current]::handleDBLDValues 
        ::kernel::TTYSattel::removeDriverHandler [namespace current]::handleUSBDriverChange
        ::kernel::JPEGSattel::removeImageHandler [namespace current]::jpegFinished
        # Nächste Version
        #TODO ::kernel::JPEGSattel::removeOverloadHandler [namespace current]::handleOverload
        #TODO Thread für normiertes Druckbild beenden
        # appsettings ohne defaults speichern
        set as2store $appsettings
        foreach {key} [dict keys $appsettings] {
            dict unset as2store $key default
        }
        srvLog [namespace current]::stop Debug "settings to store:\n$as2store"
        ::kernel::DB::setAppsettings "dekubitus" $as2store
        srvLog [namespace current] Info "App stopped."
        #}}}
    }; # proc stop 

    ##}}} init{}, start{}, stop{}

}; # namespace eval dekubitus 

set app_loaded dekubitus

