#!/bin/bash
#
# setup.sh
#
# Velobox
#
# Elementare initiale Setupschritte:
#   - Einrichtung des Service Interface über ttyUSB0
#   - Freigabe für root-Anmeldung über sshd
#   - Systemaktualisierung
#   - Installation allgemein benötigter Debian Pakete
#   - Einrichtung von vim und tclsh für 'velometrik'
#   - Hostname 'velobox<id>' gemäß Kommandozeile
#
# Manuelle Vorbereitungen:
#   - Start mit neuem raspios
#   - User 'velometrik' anlegen
#   - Tastatur 'German'
#   - ssh enable (raspi-config)
#   - root Passwort setzen
#   - Ethernet-Kabel schließen
#   - Diese Datei zu velometrik hochladen
#   Start kann als 'velometrik' erfolgen,
#   danach erfolgt Neustart als 'root'.
#
# Logfile: /var/local/log/vlbsetup.log
#
# Velometrik GmbH
#
# Historie:
#   18.02.2023  Siegmar Müller  Begonnen
#   01.03.2023  Siegmar Müller  <id> als Argument
#   01.01.2024  Siegmar Müller  oemuser
#   15.01.2024  Siegmar Müller  Laden von 32 Bit Kernel bei 32 Bit Architektur
#   16.01.2024  Siegmar Müller  sshd Freigabe für root nach apt-get upgrade
#   19.03.2024  Siegmar Müller  vlbupdate wird nach /usr/local/bin kopiert
#

# Hilfe ausgeben
help() {
    echo
    echo "Velobox Initiales Setup:"
    echo
    echo "Aufruf: $0 [<optionen>]"
    echo "Optionen:"
    echo "-i <id>   Box identifier (=> Hostname='velobox<id>')"
    echo "-?        Hilfe ausgeben und beenden"
    if [ $UID -ne 0 ]; then
        echo "(Die Ausführung erfordert root-Rechte.)"
    fi
    echo
}

# Start
echo
echo "$0 $@"
echo

BOXID="III"

# Optionen aus der Kommandozeile holen.
while getopts i:? OPTION; do
    case $OPTION in
        i) BOXID=$OPTARG;;
        ?) help; exit 0;;
    esac
done

# Ausführung als root
if [ $UID -ne 0 ]; then
    echo "Das Skript benötigt 'root' Rechte."
    su -c "$0 $*"
    exit $?
fi

### Vorbereitung für Logausgaben und Fehlertest mit eventuellem Abbruch

LOGFILE=/var/local/log/vlbsetup.log
if [ -f $LOGFILE ]; then
    echo -e "\n-------------------------------------------------------------------------------" >> $LOGFILE
else
    # Das Logverzeichnis existiert noch nicht in einem neu aufgesetzten System.
    mkdir -p `dirname $LOGFILE`
fi
echo "setup.sh `date '+%d.%m.%Y %H:%M:%S'`" | tee -a $LOGFILE

SETUPDIR="$(cd $(dirname $0) > /dev/null 2>&1; pwd -P)"
echo "SETUPDIR = $SETUP_DIR" | tee -a $LOGFILE

# Ausgabe von Logmeldungen in der Form:
#   echo "Meldung" | tee -a $LOGFILE
log() {
    echo "$*" | tee -a $LOGFILE
}

# Setup bei Fehler abbrechen
#   cancelSetup [<exit_status>]
# Ohne Angabe von <exit_status> wird mit Status 0 abgebrochen.
# Bei <exit_status> 0 geht es weiter.
cancelSetup() {
    if [ $# -eq 0 ]; then
        log "setup.sh abgebrochen."
        exit 0
    fi
    if [ $1 -ne 0 ]; then
    	log "setup.sh mit Status $1 abgebrochen"
    	exit $1
    fi
}

# Umleitung von Kommandoausgaben:
#   <command> 2>&1 | tee -a $LOGFILE
# Umleitung von Kommandoausgabe und Statusauswertung:
#   <command> 2>&1 | tee -a $LOGFILE
#   [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1

echo "Service Interface über USB einrichten ..." | tee -a $LOGFILE
if [ -L /etc/systemd/system/getty.target.wants/serial-getty\@ttyUSB0.service ]; then
    log "Das Service Interface ist bereits eingerichtet."
    log "(/etc/systemd/system/getty.target.wants/serial-getty\@ttyUSB0.service existiert.)"
else # Interface einrichten
    if [ ! -d /etc/systemd/system/getty.target.wants/ ]; then
        log "Fehler: Es muß erst ein Benutzer angelegt werden!"
        log "(Verzeichnis /etc/systemd/system/getty.target.wants/ fehlt.)"
        cancelSetup 1
    fi
    # Statt systemctl enable ...
    ln -s /lib/systemd/system/serial-getty\@.service /etc/systemd/system/getty.target.wants/serial-getty\@ttyUSB0.service 2>&1 | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    log "Das Service Interface wurde erfolgreich eingerichtet."
fi


#### Debian-Basispakete installieren

DEBIAN_PAKETE="vim anacron"
DEBIAN_PAKETE="$DEBIAN_PAKETE tcl8.6 tcl-tclreadline tcllib tcl-thread tcl-tls tcl-expect tcl-itcl4 tcl8.6-tdbc tcl8.6-tdbc-postgres tcl8.6-tdbc-sqlite3"

log "apt-get update ..."
apt-get update | sed $'s/^.*\r//' | tee -a $LOGFILE
log "apt-get upgrade ..."
apt-get upgrade -y | sed $'s/^.*\r//' | tee -a $LOGFILE
log "upgrade beendet."
for DEB_PAKET in $DEBIAN_PAKETE; do
    STATUS=`dpkg-query --showformat '${db:Status-Status}' -W $DEB_PAKET 2>&1`
    RC=$?
    if [ $RC -eq 2 ]; then
        log "FEHLER: $STATUS"
        cancelSetup 2
    fi
    if [ $RC -eq 1 ]; then
        STATUS="not-installed"
    fi
    if [ "$STATUS" == "installed" ]; then
        log "$DEB_PAKET ist bereits installiert."
    elif [ "$STATUS" == "not-installed" ]; then
        apt-get -y install $DEB_PAKET 2>&1 | tee -a $LOGFILE
    	[ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    else
        log "FEHLER: Unbekannter Status ($STATUS) für Paket $DEB_PAKET."
        cancelSetup 1
    fi
done


# sshd freigeben für root-Anmeldung
grep -q '^PermitRootLogin yes' /etc/ssh/sshd_config
if [ $? -eq 0 ]; then
    log "root login über ssh ist bereits freigegeben."
else
    log "root login über ssh wird freigegeben ...."
    cp -v /etc/ssh/sshd_config /etc/ssh/sshd_config~ 2>&1 | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] && exit 1
    sed '/#PermitRootLogin/a PermitRootLogin yes' /etc/ssh/sshd_config~ > /etc/ssh/sshd_config
    [ $? -ne 0 ] && exit 1
    service sshd restart 2>&1 | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] && exit 1
    log "... root login über ssh ist freigegeben."
fi


# vim für velometrik einrichten
cat > /home/velometrik/.vimrc <<-EOT
	syntax enable
	set foldmethod=marker
	set tabstop=4 shiftwidth=4 expandtab
    au BufNewFile,BufRead *.tcls set filetype=html
EOT
chown velometrik /home/velometrik/.vimrc
chgrp velometrik /home/velometrik/.vimrc
log ".vimrc für velometrik angelegt."

# tclsh für velometrik einrichten
cat > /home/velometrik/.tclshrc <<-EOT
	if {\$tcl_interactive} {
	    package require tclreadline
	    ::tclreadline::Loop
	}
EOT
chown velometrik /home/velometrik/.tclshrc
chgrp velometrik /home/velometrik/.tclshrc
log ".tclshrc für velometrik angelegt."


## OEM-User anlegen
useradd -s /bin/bash -G staff,dialout -d /home/oemuser -m oemuser 2>&1 | tee -a $LOGFILE
case $? in
    0)  usermod --password $(echo oempassword | openssl passwd -1 -stdin) oemuser 2>&1 | tee -a $LOGFILE
        log "oemuser wurde angelegt."
        ;;
    9)  log "oemuser existiert bereits. Passwort wird nicht geändert"
        ;;
    *)  log "oemuser konnte nicht angelegt werden."
        exit 1;;
esac

# vim für oemuser einrichten
cat > /home/oemuser/.vimrc <<-EOT
	syntax enable
	set foldmethod=marker
	set tabstop=4 shiftwidth=4 expandtab
    au BufNewFile,BufRead *.tcls set filetype=html
EOT
chown oemuser /home/oemuser/.vimrc
chgrp oemuser /home/oemuser/.vimrc
log ".vimrc für oemuser angelegt."

# tclsh für oemuser einrichten
cat > /home/oemuser/.tclshrc <<-EOT
	if {\$tcl_interactive} {
	    package require tclreadline
	    ::tclreadline::Loop
	}
EOT
chown oemuser /home/oemuser/.tclshrc
chgrp oemuser /home/oemuser/.tclshrc
log ".tclshrc für oemuser angelegt."


# velobox.conf kopieren
if [ -f /usr/local/etc/velobox.conf ]; then
    log "Existierende /usr/local/etc/velobox.conf wird beibehalten"
else
    cp -v ${SETUPDIR}/velobox.conf /usr/local/etc/ 2>&1 | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    chgrp -v staff /usr/local/etc/velobox.conf 2>&1 | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    chmod -v g+w /usr/local/etc/velobox.conf 2>&1 | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
fi


## /boot/config.txt anpassen
if [ -f /boot/config.txt.bup-velobox ]; then
    log "Originale /boot/config.txt wurde schon früher als /boot/config.txt.bup-velobox gesichert."
else
    cp -v /boot/config.txt /boot/config.txt.bup-velobox | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    log "Originale /boot/config.txt als /boot/config.txt.bup-velobox gesichert."
fi
log "Modifizierte /boot/config.txt wird aus /boot/config.txt.bup-velobox erstellt."
cat /boot/config.txt.bup-velobox - > /boot/config.txt <<-EOT

	# Inserted for VeloBox'
	[all]
	# Für VeloBox3
	# Internal Wifi conflicts internal Bluetooth => turn off WiFi
	dtoverlay=disable-wifi

EOT
[ $? -ne 0 ] &&  cancelSetup 1
if [ `dpkg --print-architecture` == "armhf" ]; then # 32-Bit Architecture
    # => Laden des 64-Bit Kern unterbinden
    echo "# Force 32 Bit Kernel" >> /boot/config.txt
    echo "arm_64bit=0" >> /boot/config.txt
fi
log "Modifizierte /boot/config.txt aus /boot/config.txt.bup-velobox erstellt.(Internes WiFi ausgeschaltet)"


### /boot/cmdline.txt anpassen
if [ ! -f /boot/cmdline.txt.bup-velobox ]; then
    cp -v /boot/cmdline.txt /boot/cmdline.txt.bup-velobox | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    log "Originale /boot/cmdline.txt gesichert."
fi
# Bonjur und Konsorten hat Probleme mit IPV6
# => IPV6 ausschalten
sed '1s/$/ ipv6.disable=1/' /boot/cmdline.txt.bup-velobox > /boot/cmdline.txt
[ $? -ne 0 ] &&  cancelSetup 1
log "Modifizierte /boot/cmdline.txt aus /boot/cmdline.txt.bup-velobox erstellt. (IPV6 ausgeschaltet)"


## vlbupdate installieren

# .tar.gz kopieren und entpacken
TARFILE=`ls $SETUPDIR/vlbupdate_*.tar.gz 2>/dev/null`
if [ `echo $TARFILE | wc -w` -ne 1 ]; then
    log "FEHLER: .tar.gz für vlbupdate fehlt oder mehrdeutig."
    cancelSetup 1
fi
cp $TARFILE /usr/local
pushd /usr/local
tar -x -f $TARFILE
popd
VLBUPDATEDIR=`echo $(basename $TARFILE) | sed 's/\.tar\.gz$//'`
rm -f /usr/local/${VLBUPDATEDIR}.tar.gz
rm -f /usr/local/vlbupdate_next
ln -s /usr/local/$VLBUPDATEDIR /usr/local/vlbupdate_next || cancelSetup 1
# Startskript kopieren und scharf schalten
cp -v $SETUPDIR/vlbupdate.sh /etc/init.d/ 2>&1  | tee -a $LOGFILE
[ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1
update-rc.d vlbupdate.sh defaults 2>&1  | tee -a $LOGFILE
[ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1
log "Service vlbupdate.sh wird gestartet ..."
service vlbupdate.sh restart 2>&1  | tee -a $LOGFILE
[ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1

# vlbupdate über $PATH zugänglich machen
if [ -x /usr/local/$VLBUPDATEDIR/bin/`uname -m`/vlbupdate ]; then
    cp -v /usr/local/$VLBUPDATEDIR/bin/`uname -m`/vlbupdate /usr/local/bin 2>&1 | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    
else
    log "/usr/local/$VLBUPDATEDIR/bin/`uname -m`/vlbupdate fehlt."
    cancelSetup 1
fi

## vlbupdate installiert


# Hostname setzen, falls noch nicht geschehen oder explizit (Option -i) angefordert
hostname=`cat /etc/hostname`
if [ ${hostname:0:7} == "velobox" ] && [ "$BOXID" == "III" ]; then
    log "Hostname $hostname wird beibehalten."
else
    echo "velobox${BOXID}" > /etc/hostname
    # In /etc/hosts steht noch raspberrypi als Synonym für localhost.
    log "Hostname 'velobox${BOXID}' gesetzt => reboot"
    echo "Neuer Name für avahi-resolve 'velobox${BOXID}.local'"
    read -p "Reboot? [Yn]" A
    A=${A:-Y}
    [ ${A^^} == "Y" ] && reboot
fi

log "setup erfolgreich beendet."

