# Kommandos für vlbupdate der VeloboxIII
#
# Aufruf von der ws Domäne vlbupdate
#
# Das Hauptskript holt diese Kommandos
# in den namespace ::vlbupdate.
#

# Historie:
# 10.07.2023 Siegmar Müller Begonnen
# 27.07.2023 Siegmar Müller Ermittlung der verfügbaren Updates fertig
# 04.01.2024 Siegmar Müller Teilweise Umgehung des Abstuzproblems bei queryInfos
#


srvLog commands.tcl Debug "Load commands.tcl ..."

package require json
package require http
package require tls

::http::register https 443 ::tls::socket

# Workaround für Fehler in package http 2.9.5
# (Der Interpreter stürzt ab, wenn der Server bei ::http::geturl nicht erreichbar ist.)
#
# Prüft, ob der angegebene Server auf Port 80 lauscht.
# Ist der Server nicht erreichbar, wird eine Warnung ins Logfile geschrieben.
# @param server <protocol>://<servername>
# @return       1 erreichbar, 0 nicht erreichbar
proc serviceAvailableOn {server} {; #{{{
    if {[catch {
        srvLog [lindex [info level 0] 0] Debug "serviceAvailableOn $server ?"
        set sock [socket $server 80]
        srvLog [lindex [info level 0] 0] Debug "serviceAvailableOn $server !"
        close $sock
        } msg]} {
    } {
        srvLog [lindex [info level 0] 0] Warn "$msg"
        return 0
    }   
    return 1
    #}}}
}; # proc serviceAvailableOn 


# Liegt die angegebene Version im angegebenen Bereich ?
# @param version        Interessierende Version
# @param version_min    Unteres Ende des Versionsbereichs
# @param version_max    Oberes Ende des Versionsbereichs
# @return 1 ja, 0 nein
# @throws error bei ungültigem Versionsstring
proc versionBetween {version version_min version_max} {; #{{{
    # Versionsstrings syntaktisch prüfen
    foreach v [list $version $version_min $version_max] {
        if {![regexp {^[0-9]+\.[0-9]+\.[0-9]+$} $v]} {
            error "invalid_version_string \"$v\""
        }
    }
    set mmr  [split [regsub {_.*$} $version ""] "."]
    set mmr_min  [split $version_min "."]
    set mmr_max  [split $version_max "."]
    for {set i 0} {$i<3} {incr i} {
        if {[lindex $mmr $i] < [lindex $mmr_min $i]} {
            return 0
        }
        if {[lindex $mmr $i] > [lindex $mmr_max $i]} {
            return 0
        }
    }
    return 1
    #}}}
}; # proc versionBetween 


# Prüfen, ob unter den gegeben Umständen ein Update (Setup) ausgeführt werden sollte
# Ein Update/Setup erfolgt:
#   wenn es sich um eine Erstinstallation handlet und noch nichts installiert wurde.
#   wenn eine Vorabversion der Zielversion installiert ist.
#   wenn die installierte Version im geforderten Versionsbereich liegt.
# @param version        Installierte Version oder Leerstring falls nicht installiert
# @param version_min    Älteste vorausgesetzte Version für Update
# @param version_max    Späteste vorausgesetzte Version für Update
# @param version_to     Zielversion
# @return               1 ja, 0 nein
proc shouldUpdate {version version_min version_max version_to} {; #{{{
    if {"$version_min" == "" && "$version_max" == ""} {; # Erstinstallation
        if {"$version" == ""} {; # Erstinstallation erforderlich
            # => Geht in Ordnung
            return 1
        } else {; # nicht anwendbar
            return 0
        }
    }
    if {[regexp {(^[0-9.]+)_(.*$)} $version match plain ancestor]} {; # Vorabversion ist installiert
        if {$plain == $version_to} {; # auf fertige Version aktualisieren
            return 1
        }
        # Ansonsten erfolgt Update wie bei allen anderen (fertigen) Versionen
        set version $plain
    }
    if {[catch {set should_update [versionBetween $version $version_min $version_max]} msg]} {
        srvLog [lindex [info level 0] 0] Error "$msg => no update"
        srvLog [lindex [info level 0] 0] Debug "\"$version\" \"$version_min\" \"$version_max\" \"$version_to\""
        ::WSServer::disposeServerMessage vlbupdate text "{\"wsevent\": \"error\", \"error\": \"internal\", \"message\": \"See log\"}"
        return 0
    }
    return $should_update
    #}}}
}; # proc shouldUpdate 


# Sonderzeichen in einem String gemäß JSON-Definition ersetzen
# @param string Beliebiger String
# @return   String mit ersetzten Sonderzeichen
proc jsonEscape {string} {; #{{{
    set json ""
    foreach {char} [split $string ""] {
        switch $char {
            "\"" {
                append json "\\\""
            }
            "\\" {
                append json "\\\\"
            }
            "\/" {
                append json "\\/"
            }
            "\\" {
                append json "\\\\"
            }
            "\b" {
                append json "\\b"
            }
            "\f" {
                append json "\\f"
            }
            "\n" {
                append json "\\r"
            }
            "\r" {
                append json "\\n"
            }
            "\t" {
                append json "\\t"
            }
            default {
                set code [scan $char %c]
                if {$code > 127} {
                    append json [format "\\u%04x" $code]
                } else {
                    append json $char
                }

            }
        }
    }
    return $json
    #}}}
}; # proc jsonEscape 


# Abfrage der infos-Domäne des vmkstationd
# @return   infos JSON Objekt wie von vmkstationd geliefert
#                 Leerstring, wenn vmkstationd nicht geantwortet hat
proc queryInfos {} {; #{{{
    #TODO vmkstationd kann runtergefahren oder gar nicht installiert sein.
    #     -> Dann bringt geturl den Interpreter aus ungeklärter Ursache zum Absturz.
    #        (selbiges bei socket oder irgend einem exec)
    # Workaround für den Fall, daß vmkstationd nicht installiert ist:
    if {![file exists /usr/local/etc/vmkstationd.conf]} {
        return ""
    }
    # Das wird vom selben Rechner geholt. => Es kann gewartet werden.
    if {[catch {::http::geturl http://localhost/infos} token]} {
        srvLog [lindex [info level 0] 0] Error "$token"
        return ""
    }
    ::http::wait $token
    if {[set status [::http::status $token]] == "ok"} {
        set infos [::http::data $token]
    } else {
        srvLog [lindex [info level 0] 0] Error "$status"
        set infos ""
    }
    ::http::cleanup $token
    return $infos
    #}}}
}; # proc queryInfos 


# Die auf dem Server verfügbaren Updates sind angekommen
# Die anzuwendenden Updates werden jetzt anhand der aktuellen Installation (queryUpdates)
# ausgewählt, vorgeschlagen und als wsevent "updates" an die Clients geschickt.
# Sofern der Server einen Fehler meldet, wird dieser zurückgegeben (wsevent "error").
# @param dict_infos /infos vom vmkstationd als dict
#                   oder nur velobox.conf wie bei /infos mit Schlüssel "velobox"
# @param token  Token des HTTP-Kanals
proc updatesArrived {dict_infos token} {; #{{{
    upvar #0 $token state

    set updates [list]
    srvLog [lindex [info level 0] 0] Info "Status: $state(status) Code: [::http::ncode $token]"
    if {$state(status) == "ok" && [::http::ncode $token] == 200} {; #{{{
        # Damit muß das nicht ok sein => unter http nachsehen
        if {[catch {
            # Daten übernehmen
            set vlbupdates [::http::data $token]
            if {[catch {set dict_vlbupdates [::json::json2dict $vlbupdates]} result]} {
                # Das kann nur passieren, wenn die Datei auf dem Server fehlerhaft ist, und wird deshalb
                # als quasi interner Fehler wieder aufgefangen.
                error "Invalid vlbupdates.json: " "[string range $vlbupdates 0 40] ..."
            }
            # Relevante Updates heraussuchen
            set update_velobox [dict create]
            set update_vmkstationd [dict create]
            set update_apps [dict create]
            set update_htdocs [dict create]
            set LANG [dict get $dict_infos velobox LANG]
            ## Updates für velobox und vmkstationd
            # (vmkstationd.VERSION fehlt in dict_infos, wenn vmkstationd nicht reagiert hat.)
            foreach level {velobox vmkstationd} {
                # Aktuelle Version ohne Ergänzungen übernehmen
                if {[dict exists $dict_infos $level VERSION]} {
                    regexp {[0-9]+\.[0-9]+\.[0-9]+} [dict get $dict_infos $level VERSION] VERSION
                } else {; # nicht installiert oder gestoppt
                    set VERSION ""
                }
                foreach update [dict get $dict_vlbupdates $level] {
                    if {[shouldUpdate $VERSION [dict get $update FROM_MIN] [dict get $update FROM_MAX] [dict get $update TO]]} {
                        #srvLog [lindex [info level 0] 0] Debug "$level: $update"
                        if {[dict exists $update comment $LANG]} {
                            set comment [dict get $update comment $LANG]
                        } else {
                            set comment [dict get $update comment en]
                        }
                        # Sonderzeichen schon hier in JSON umsetzen
                        set comment [jsonEscape $comment]
                        lappend updates [dict create software $level file [dict get $update file] TO [dict get $update TO] comment $comment]
                        break
                    }
                }
            }
            srvLog [lindex [info level 0] 0] Debug "vmkstationd.status: [dict get $dict_infos vmkstationd status]"
            # Schluß, wenn vmkstationd gestoppt
            if {"[dict get $dict_infos vmkstationd status]" == "stopped"} {
                srvLog [lindex [info level 0] 0] Warn "vmkstationd stopped."
            }
            set USAGE [dict get $dict_infos velobox USAGE]
            # Wenn USAGE für die Velobox noch nicht festgelegt wurde, ist hier Schluß.
            if {"$USAGE" != ""} {; #{{{
                ## Updates für apps gemäß USAGE
                dict for {appname specification} [dict get $dict_vlbupdates apps] {; #{{{
                        if {!($USAGE in [dict get $specification usage])} {
                            srvLog [lindex [info level 0] 0] Debug "$appname is not for $USAGE"
                            continue
                        }
                    # Aktuelle Version von appname
                    if {[dict exists $dict_infos apps $appname]} {
                        set appversion [dict get $dict_infos apps $appname VERSION]
                    } else {; # App ist noch nicht installiert.
                        set appversion ""
                    }
                    foreach {version} [dict get $specification versions] {
                        srvLog [lindex [info level 0] 0] Debug "$appname FROM '$appversion' TO '[dict get $version TO]' ?"
                        if {[dict exists $version comment $LANG]} {
                            set comment [dict get $version comment $LANG]
                        } else {
                            set comment [dict get $version comment en]
                        }
                        # Sonderzeichen schon hier in JSON umsetzen
                        set comment [jsonEscape $comment]
#                        #??? Das soll shouldAppdate mit behandeln:
#                        if {"[dict get $version FROM_MIN][dict get $version FROM_MAX]" == ""} {; # neue App
#                            if {"$appversion" != ""} {; # App ist bereits installiert.
#                                # => nicht nochmal installieren
#                                continue
#                            }
#                            # App ist noch nicht installiert.
#                            lappend updates [dict create software $appname file [dict get $version file] TO [dict get $version TO] comment $comment]
#                            break
#                        } else {
#                            # Versionsvergleich
                            if {[shouldUpdate $appversion [dict get $version FROM_MIN] [dict get $version FROM_MAX] [dict get $version TO]]} {
                                lappend updates [dict create software $appname file [dict get $version file] TO [dict get $version TO] comment $comment]
                                break
                            }
#                        }
                    }
                    #}}}
                }; # Apps durchgehen
    
                set IMPLEMENTATION [dict get $dict_infos velobox IMPLEMENTATION]
                # Wenn implementation noch nicht festgelegt ist, ist hier Schluß.
                if {"$IMPLEMENTATION" != ""} {; #{{{
                    ## Updates für apps gemäß USAGE und IMPLEMENTATION
                    srvLog [lindex [info level 0] 0] Debug "Checking htdocs for $USAGE/$IMPLEMENTATION"
                    # Aktuelle Version von htdocs
                    if {[dict exists $dict_infos htdocs]} {
                        set VERSION [dict get $dict_infos htdocs VERSION]
                        set htdocs_usage [dict get $dict_infos htdocs usage]
                        set htdocs_implementation [dict get $dict_infos htdocs implementation]
                    } else {
                        set VERSION ""
                        set htdocs_usage $USAGE
                        set htdocs_implementation $IMPLEMENTATION
                    }
                    foreach htdocs [dict get $dict_vlbupdates htdocs] {; #{{{ htdocs für alle Anwendungen durchgehen
                        if {"[dict get $htdocs usage]" != "$USAGE" || "[dict get $htdocs implementation]" != "$IMPLEMENTATION"} {
                            srvLog [lindex [info level 0] 0] Debug "htdocs for [dict get $htdocs usage]/[dict get $htdocs implementation] ignored."
                            continue
                        }
                        # Versionen durchgehen
                        foreach version [dict get $htdocs versions] {
                            if {[dict exists $version comment $LANG]} {
                                set comment [dict get $version comment $LANG]
                            } else {
                                set comment [dict get $version comment en]
                            }
                            # Sonderzeichen schon hier in JSON umsetzen
                            set comment [jsonEscape $comment]
                            # Diese htdocs installieren, wenn ...
                            #   ... sie für Erstinstallation sind und die installierten htdocs
                            #   eine andere Spezifikation (usage/implementation) haben als die Velobox.
                            if {"[dict get $version FROM_MIN]" == "" && "[dict get $version FROM_MAX]" == ""} {; # Erstinstallation
                                if {"$htdocs_usage" != "$USAGE" || "$htdocs_implementation" != "$IMPLEMENTATION"} {
                                    # Andere Spezifikation => installieren
                                    lappend updates [dict create software htdocs file [dict get $version file] TO [dict get $version TO] comment $comment]
                                    break
                                }
                            }
                            #   ... sie ein Update für die installierten htdocs sind.
                            if {[shouldUpdate $VERSION [dict get $version FROM_MIN] [dict get $version FROM_MAX] [dict get $version TO]]} {
                                lappend updates [dict create software htdocs file [dict get $version file] TO [dict get $version TO] comment $comment]
                                break
                            }
                        }; # Versionen durchgehen
                        #}}}
                    }; # htdocs für alle Anwendungen durchgehen
                #}}}
                }; # if $IMPLEMENTATION spezifiziert
                #}}}
            }; # if $USAGE spezifiziert

            # Herausgesuchte Updates (Liste updates) als JSON Array über Websocket ausliefern
            #srvLog [lindex [info level 0] 0] Debug "$updates"
            set json_updates ""
            set sep ""
            foreach update $updates {
                append json_updates "${sep}{"
                set sepp ""
                dict for {key value} $update {
                    append json_updates "${sepp}\"$key\": \"$value\""
                    set sepp ", "
                }
                append json_updates "}"
                set sep ", "
            }
            ::WSServer::disposeServerMessage vlbupdate text "{\"wsevent\": \"updates\", \"updates\": \[$json_updates\]}"
        } result]} {; # Die Übernahme der Updates ist irgendwo gescheitert, wo das nicht hätte sein dürfen.
            global errorInfo

            srvLog [lindex [info level 0] 0] Error "$result\n$errorInfo"
            ::WSServer::disposeServerMessage vlbupdate text "{\"wsevent\": \"error\", \"error\": \"internal\", \"message\": \"See log\"}"
        }
        #}}}
    } else {; #{{{ $state(status) != "ok" oder HTTP code != 200
        # Fehler analysieren
        if {$state(status) == "ok"} {
            set msg "server responds: [::http::code $token]"
        } else {
            set msg "status = $state(status)"
        }
        srvLog [lindex [info level 0] 0] Error "cant_query_updates: $msg"
        ::WSServer::disposeServerMessage vlbupdate text "{\"wsevent\": \"error\", \"error\": \"cant_query_updates\", \"message\": \"$msg\"}"
        #}}}
    }
    ::http::cleanup $token
#}}}
}; # proc updatesArrived 


# Updates für diese Velobox beim Server abfragen
# und als wsevent "updates" an die Websocket Clients schicken.
# Infos von vmkstationd als wsevent "infos" an die Clients schicken.
# Diese Prozedur startet die Abfrage, die von proc updatesArrived beendet wird.
proc queryUpdates {} {; #{{{
    set infos [queryInfos]
    if {"$infos" == ""} {; #{{{ Fehler auswerten, melden, infos aus velobox.conf bauen
        # dict_infos aus velobox.conf bauen
        set velobox [dict create LANG en USAGE "" IMPLEMENTATION ""]
            if {[file exists /usr/local/etc/velobox.conf]} {; # Konfiguration einlesen
                set fd [open /usr/local/etc/velobox.conf r]
                    while {[gets $fd line] >= 0} {
                        if {[regexp {^([A-Za-z0-9]+) *= *([^ ]+)} $line match key value]} {
                            # (Erst 'mal keine Sonderbehandlung für HARDWARE)
                            dict set velobox $key $value
                        }
                    }
                close $fd
            }
        set dict_infos [dict create velobox $velobox]
        # Solange wir es nicht besser wissen: Irgend eine (fast) unmögliche Fehlerursache
        set wsmsg "{\"wsevent\": \"warning\", \"warning\": \"no_infos\", \"reason\": \"unknown\"}"
        dict set dict_infos vmkstationd status unknown
        if {![file isdirectory /usr/local/vmkstationd]} {
            # vmkstationd ist nicht installiert
            set wsmsg "{\"wsevent\": \"warning\", \"warning\": \"no_infos\", \"reason\": \"not_installed\"}"
                dict set dict_infos vmkstationd status not_installed
        } elseif {[catch {exec service vmkstationd.sh status} result status]} {
            set errorcode [dict get $status -errorcode]
                if {"[lindex $errorcode 0]" == "CHILDSTATUS"} {
                    set status [lindex $errorcode 2]
                        switch $status {
                            3 {
                                # vmkstationd ist angehalten (nicht gestartet)
                                set wsmsg "{\"wsevent\": \"warning\", \"warning\": \"no_infos\", \"reason\": \"stopped\"}"
                                    dict set dict_infos vmkstationd status stopped
                            }
                            4 {
                                # /usr/local/vmkstationd ist vorhanden aber kein sysstemd kennt kein Startskript
                                set wsmsg "{\"wsevent\": \"warning\", \"warning\": \"no_infos\", \"reason\": \"not_installed\"}"
                                    dict set dict_infos vmkstationd status not_installed
                            }
                        }
                }
        }
        ::WSServer::disposeServerMessage vlbupdate text $wsmsg
        #}}}
    } else {; # queryInfos war erfolgreich.
        ::WSServer::disposeServerMessage vlbupdate text "{\"wsevent\": \"infos\", \"infos\": $infos}"
        set dict_infos [::json::json2dict $infos]
        dict set dict_infos vmkstationd status running
    }
    # Downloadserver konfiguriert ?
    if {!([dict exists $dict_infos velobox DOWNLOADSERVER] && [dict exists $dict_infos velobox UPDATEDIR])} {
        set msg "DOWNLOADSERVER and/or UPDATEDIR not configured"
        srvLog [lindex [info level 0] 0] Error "$msg"
        ::WSServer::disposeServerMessage vlbupdate text "{\"wsevent\": \"error\", \"error\": \"cant_query_updates\", \"message\": \"$msg\"}"
        return
    }
    # Dowloadserver erreichbar ?
    if {![serviceAvailableOn "[dict get $dict_infos velobox DOWNLOADSERVER]"]} {
        set msg "Can't contact [dict get $dict_infos velobox DOWNLOADSERVER]"
        srvLog [lindex [info level 0] 0] Error "$msg"
        ::WSServer::disposeServerMessage vlbupdate text "{\"wsevent\": \"error\", \"error\": \"cant_query_updates\", \"message\": \"$msg\"}"
        return
    }
    
    set content_url "[dict get $dict_infos velobox DOWNLOADSERVER][dict get $dict_infos velobox UPDATEDIR]/vlbupdates.json"
    srvLog [lindex [info level 0] 0] Info "Fetching $content_url"
    if {[catch {::http::geturl $content_url -command "[namespace current]::updatesArrived {$dict_infos}"} token]} {
        srvLog [lindex [info level 0] 0] Error "$token"
        ::WSServer::disposeServerMessage vlbupdate text "{\"wsevent\": \"error\", \"error\": \"cant_query_updates\", \"message\": \"$token\"}"
    }
    #}}}
}; # proc queryUpdates 


# Irgendetwas abfragen
# Syntax:   query <what>
#           <what>: updates
# @return   Leerstring oder Fehler
proc doQuery {args} {; #{{{
    if {[llength $args] == 1} {
        switch [lindex $args 0] {
            "updates" {
                queryUpdates
                return ""
            }
        }
    }
    return "Must be: query updates"
    #}}}
}; # proc doQuery 


# Ein Update mit vlbupdate ausführen
# Syntax:   update <download-file>
# @return   Leerstring oder Fehler
proc doUpdate {args} {; #{{{
    if {[llength $args] != 1} {
        return "Must be: update <download-file>"
    }
    set filename [lindex $args 0]
    # Startmeldung
    ::WSServer::disposeServerMessage vlbupdate text "{\"wsevent\": \"updating\", \"file\": \"$filename\"}"
    if {[catch {exec ${::BINDIR}/vlbupdate $filename >>& /var/local/log/vlbupdate.log} msg]} {
        return "$msg (check logfile)"
    }
    # Fertigmeldung
    ::WSServer::disposeServerMessage vlbupdate text "{\"wsevent\": \"updated\", \"file\": \"$filename\"}"
    return ""
    #}}}
}; # proc doUpdate 


# Eingegangenes Kommando an die Kommandoprozedur weiterleiten
# (Aufruf von custom/wsserver/wsdomains/vlbupdate.tcl)
# Bei Fehlern geht eine Meldung an die Domäne vlbupdate.
# @param command    Kommando wie vom Client eingegangen
proc handleCommand {command} {; #{{{
    srvLog [lindex [info level 0] 0] Debug "Command received: $command"
    set commandname [regsub {  *.*$} $command ""]
    set args [regsub {[a-zA-Z][a-zA-Z0-9]* *} $command ""]
    set result ""

    if {[catch {
        switch $commandname {
            "query" {
                set result [doQuery {*}$args]
            }
            "update" {
                set result [doUpdate {*}$args]
            }
            default {
                set result "Unknown command name: $commandname"
            }
        }
    } error_msg]} {; # interner Fehler
        if {[info exists errorInfo]} {
            append error_msg "\n$errorInfo"
        }
        ::WSServer::disposeServerMessage vlbupdate text "{\"wsevent\": \"error\", \"error\": \"internal\", \"message\": \"[jsonEscape $error_msg]\"}"
        return
    }
    if {"$result" != ""} {
        ::WSServer::disposeServerMessage vlbupdate text "{\"wsevent\": \"error\", \"error\": \"client\", \"message\": \"[jsonEscape $result]\"}"
    } else {
        srvLog [lindex [info level 0] 0] Info "Command executed: $command"
    }
    #}}}
}; # proc handleCommand 

srvLog commands.tcl Debug "commands.tcl loaded."

