#!/bin/bash
#
# .../velobox/setup/setup_RB-TFT-3.5.sh
# Setup für Displayeinheit
# (Funktioniert bisher nur bei 32 Bit.)
#
# Velometrik GmbH
#
#  Historie:
# 10.03.2023 Siegmar Müller Begonnen
# 14.03.2023 Siegmar Müller Display funktioniert
# 09.10.2023 Siegmar Müller Fehlendes apt-get update eingefügt
#   TODO Touchfunktion funktionstüchtig machen

# Voller Pfad des Startverz. mit '/' am Ende
SETUP_DIR="$(cd $(dirname $0) > /dev/null 2>&1; pwd -P)/"

LOGFILE=/var/local/log/vlbsetup.log
if [ ! -d `dirname $LOGFILE` ]; then
    echo "Das Verzeichnis /var/local/log/ existiert nicht."
    echo "Wurde setup.sh ausgeführt?"
    exit 1
fi

# Hilfe ausgeben
help() {
    echo
    echo "Velobox Einrichtung des 3.5'' TFT Displays:"
    echo
    echo "Aufruf: $0 [<optionen>]"
    echo "Optionen:"
    echo "-?        Hilfe ausgeben und beenden"
    if [ $UID -ne 0 ]; then
        echo "(Die Ausführung erfordert root-Rechte.)"
    fi
    echo
}

# Optionen aus der Kommandozeile holen.
while getopts ? OPTION; do
    case $OPTION in
        ?) help; exit 0;;
    esac
done

if [ `uname -m ` != "armv7l" ]; then
    echo "Das Setup funktioniert derzeit nur auf der armv7l 32-Bit Architektur."
    exit 1
fi

# Setup nur auf 32 Bit
if [ `uname -m` != "armv7l" ]; then
    echo "Das TFT-Display kann nur auf 32 Bit (armv7l) installiert werden."
    exit 1
fi

# Ausführung als root
if [ $UID -ne 0 ]; then
    echo "Das Skript benötigt 'root' Rechte."
    su -c "$0 $*"
    exit $?
fi

if [ -f $LOGFILE ]; then
    echo -e "\n-------------------------------------------------------------------------------" >> $LOGFILE
else
    # Das Logverzeichnis existiert noch nicht in einem neu aufgesetzten System.
    mkdir -p `dirname $LOGFILE`
fi
# Start
echo
echo "$0 `date '+%d.%m.%Y %H:%M:%S'`" | tee -a $LOGFILE
echo

# Ausgabe von Logmeldungen in der Form:
#   echo "Meldung" | tee -a $LOGFILE
log() {
    echo "$*" | tee -a $LOGFILE
}

# Setup bei Fehler abbrechen
#   cancelSetup [<exit_status>]
# Ohne Angabe von <exit_status> wird mit Status 0 abgebrochen.
# Bei <exit_status> 0 geht es weiter.
cancelSetup() {
    if [ $# -eq 0 ]; then
        log "$0 abgebrochen."
        exit 0
    fi
    if [ $1 -ne 0 ]; then
    	log "$0 mit Status $1 abgebrochen"
    	exit $1
    fi
} # cancelSetup()


# Installierte Hardware in velobox.conf eintragen
# Aufruf: updateConf <hardware>
updateConf() {
    if [ ! -f /usr/local/etc/velobox.conf ]; then
        echo "/usr/local/etc/velobox.conf existiert nicht => Eintrag der Hardware ist unmöglich." 
        return 1
    fi
    if `grep -q '^HARDWARE=.*'$1 /usr/local/etc/velobox.conf`; then
        echo "$1 ist bereits als Hardwarekomponente in /usr/local/etc/velobox.conf eingetragen."
    else
        # <hardware> ist nicht eingetragen
        if `grep -q '^HARDWARE=' /usr/local/etc/velobox.conf`; then # $HARDWARE muß modifiziert werden
            cp -v /usr/local/etc/velobox.conf /usr/local/etc/velobox.conf~
            sed '/^HARDWARE=[^ ]/s/$/,'$1'/;/^HARDWARE= *$/s/= */='$1'/' /usr/local/etc/velobox.conf~ > /usr/local/etc/velobox.conf
        else # $HARDWARE ist (merkwürdigerweise) gar nicht definiert.
            # => anhängen
            echo -e "\nHARDWARE=$1" >> /usr/local/etc/velobox.conf
        fi
        echo "$1 als Hardwarekomponente in /usr/local/etc/velobox.conf eingetragen."
    fi
} # updateConf()


log "Fehlende DEBIAN Pakete nachinstallieren ..."
apt-get update 2>&1 | tee -a $LOGFILE
for DBPKG in xinit xserver-xorg-video-fbturbo xserver-xorg-input-evdev x11-xserver-utils xinput-calibrator; do
    if `dpkg -l $DBPKG 2>&1 > /dev/null`; then
        log "$DBPKG ist bereits installiert."
    else
        log "$DBPKG muß installiert werden."
        apt-get -y install $DBPKG 2>&1 | tee -a $LOGFILE
    	[ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    fi
done

# Driver Overlay übernehmen
if `cmp -s /boot/overlays/joy-IT-Display-Driver-35a-overlay.dtbo ${SETUP_DIR}joy-IT-Display-Driver-35a-overlay.dtbo`; then
    log "joy-IT-Display-Driver-35a-overlay.dtbo ist aktuell."
else # Datei kopieren
    cp -v ${SETUP_DIR}joy-IT-Display-Driver-35a-overlay.dtbo /boot/overlays/joy-IT-Display-Driver-35a-overlay.dtbo | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
fi

### /boot/config.txt anpassen
if [ ! -f /boot/config.txt.bup-RB-TFT-3.5 ]; then
    cp -v /boot/config.txt /boot/config.txt.bup-RB-TFT-3.5 | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    log "Originale /boot/config.txt als /boot/config.txt.bup-RB-TFT-3.5 gesichert."
fi
log "Modifizierte /boot/config.txt wird aus /boot/config.txt.bup-RB-TFT-3.5 erstellt."
cat /boot/config.txt.bup-RB-TFT-3.5 - > /boot/config.txt <<-EOT

	# Inserted for VeloBox' display
	[all]
	dtparam=spi=on
	# Für Testaufbau:
	#dtoverlay=joy-IT-Display-Driver-35a-overlay:rotate=90,swapxy=0
	# Für Hockereinbau:
	dtoverlay=joy-IT-Display-Driver-35a-overlay:rotate=270,swapxy=1
	dtparam=audio=on
	max_usb_current=1
	hdmi_force_hotplug=1
	config_hdmi_boost=7
	hdmi_drive=1
	hdmi_ignore_edid=0xa5000080
	
	[pi 4]
	max_framebuffers=2

EOT
[ $? -ne 0 ] &&  cancelSetup 1
log "Modifizierte /boot/config.txt aus /boot/config.txt.bup-RB-TFT-3.5 erstellt."

### /boot/cmdline.txt anpassen
if [ ! -f /boot/cmdline.txt.bup-RB-TFT-3.5 ]; then
    cp -v /boot/cmdline.txt /boot/cmdline.txt.bup-RB-TFT-3.5 | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    log "Originale /boot/cmdline.txt gesichert."
fi
# Der Eintrag console=tty1 bewirkt die Ausgabe der Logmeldungen beim Systemstart
# => Entfernen verhindert das.
sed '1{s/$/ fbcon=map:10/;s/ console=tty1//}' /boot/cmdline.txt.bup-RB-TFT-3.5 > /boot/cmdline.txt
[ $? -ne 0 ] &&  cancelSetup 1
log "/boot/cmdline.txt angepasst."

log "Beim nächsten reboot sollte das Display als Anmelde-Konsole funktionieren."

# /usr/share/X11/xorg.conf.d/99-fbturbo.conf übernehmen
if [ ! -f /usr/share/X11/xorg.conf.d/99-fbturbo.conf ]; then
    # fbdev fb0 bei rpi3, fb1 bei rpi4
    FBDEV=fb0   # rpi3
    [ -c /dev/fb1 ] && FBDEV=fb1    # rpi4
    sed "s/fb[01]/${FBDEV}/" ${SETUP_DIR}/99-fbturbo.conf > /usr/share/X11/xorg.conf.d/99-fbturbo.conf
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    log "99-fbturbo.conf kopiert (fbdev = $FBDEV)."
fi

log "Beim nächsten reboot sollte das Display mit X11 funktionieren."

# /usr/share/X11/xorg.conf.d/99-calibration.conf übernehmen
if [ ! -f /usr/share/X11/xorg.conf.d/99-calibration.conf ]; then
    cp -v ${SETUP_DIR}/99-calibration.conf /usr/share/X11/xorg.conf.d/ | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    log "99-calibration.conf kopiert."
fi

# /usr/share/X11/xorg.conf.d/10-evdev.conf übernehmen
if [ ! -f /usr/share/X11/xorg.conf.d/45-evdev.conf ]; then
    #TODO Aus der joy-it Anleitung: Unklar, was das soll:
    cp -v /usr/share/X11/xorg.conf.d/10-evdev.conf /usr/share/X11/xorg.conf.d/45-evdev.conf
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    log "45-evdev.conf kopiert."
fi

# Eintrag in velobox.conf
updateConf JIT35 || exit 1

log "Setup erfolgreich beendet."

