#
# HTTP/WS-Server für Gerätesteuerungen
#
# Author: Siegmar Müller, Velometrik GmbH
#
# Der Server geht auf Ideen aus dem
# Simple Sample httpd/1.[01] server von
# Stephen Uhler (c) 1996-1997 Sun Microsystems
# und dem
# Minimalist Websocket Example Server von
# Bezoar (a.k.a. Carl Jolly) zurück.

# Allgemeine Beschreibung
# =======================
# Der hier im namespace ::WSServer enthaltene Kode ist die Basis für spezielle
# Gerätesteuerungen im Intranet.
# Er kann als einfacher HTTP- und gleichzeitig als WS-Server arbeiten.
# Die ankommenden Clientrequests gehen in die TCL-Eventloop, um danach
# sequentiell abgearbeitet zu werden. Eine Parallelisierung findet nicht statt,
# weil i.a. mit nur einem Client zur rechnen ist. Mehrere Clients sind möglich,
# arbeiten i.a. aber nicht unabhängig voneinander. Ein Client wird parallel
# zu einer HTTP-Verbindung meist auch eine oder mehrere WS-Verbindungen nutzen.
# Der Server arbeitet mit "Domänen" ähnlich denen im tclhttp-Server von Brent Welch.
# Das sind Namen, die in der URL als erstes Wort nach dem Server stehen.
# Sie sind von vornherein als HTTP- oder WS-Domänen definiert und werden bereits beim
# Requesteingang entsprechend unterschiedlich behandelt. Insbesondere wird beim Zugriff
# auf eine WS-Domäne ein Websocket-Upgrade versucht.
# Die Domänen müssen mittels spezieller Prozeduren realisiert werden.
# Darüber hinaus können wie bei jedem gewöhnlichen Webserver mit HTTP beliebige
# Dateien abgefufen werden.
# Dateien mit der Endung ".tcls" werden als TCL-Strings interpretiert, d.h. vor ihrer
# Weitergabe durchlaufen sie den TCL-Befehl "subst", wobei Variable ersetzt und TCL-Kommandos
# ausgeführt werden. Anders als bei Apache Rivet wird hier die "harte" TCL-Tour gefahren.
# (Details dazu s. proc httpRespondTCLS)
# Der Eisatz als HTTP|HTTPS-Proxy wurde mit einer HTTP-Domäne realisiert.
# (custom/wsserver/httpdomains/stationsproxy.tcl)
#
# Zum Benutzen muß diese Datei mit 'source' in die Anwendung geholt werden.
# Von den im dann zur Verfügung stehenden Namespace ::WSServer enthaltenen Prozeduren
# sind folgende wichtig:
# ::WSServer::start zum Starten des Servers
# ::WSServer::disposeServerMessage zum Verteilen von Mitteilungen an WS-Clients
# Die Aufrufkonventionen finden sich bei den Prozedurdefinitionen.
# Der Kode enthält die HTTP-Testdomänen /header und /post, sowie
# die WS-Testdomäne /logme, die als Vorlage für eigene Domänen verwendbar sind.
# Eigene Domänen sollten in externen Dateien (eine Datei je Domäne) implementiert werden.
# Dafür ist die folgende Verzeichnisstruktur vorgesehen:
# .../<anwendung>.tcl
# .../lib/wsserver.tcl
# .../custom/wsserver/httpdomains/<domain1>.tcl
# ...                            /<domain2>.tcl
# ...                            ...
# .../custom/wsserver/wsdomains/<domain1>.tcl
# ...                          /<domain2>.tcl
# ...                          ...
# Die Anwendung muß .../lib/wsserver.tcl "sourcen", der sich seinerseits die Domänen holt.

# Hinweise:
# Der Server benutzt die in .../start/utils.tcl definierte Logprozedur srvLog.
# Mit "Clientsockets" sind hier im Kode die vom Server gelieferten Sockets
# für die Verbindungen mit den Clients gemeint. (Nicht mit socket erzeugt Clientsockets)
# Wenn in einer HTTP-Domäne die Variable REQUESTLOGLEVEL gesetzt ist,
# wird die Fertigmeldung mit diesen Loglevel geschrieben.
# Wenn REQUESTLOGLEVEL nicht gesetzt ist, wird es beim Laden auf "Info" gesetzt.
# Ein eingehendes Request, das den GET-Parameter "rnd=<zahl>" enthält wird mit
# Loglevel Debug statt Info geloggt.

#TODO puts $clientsocket ... führt gelegentlich zu ERROR: broken pipe, was von bgerror abgefangen wird
# Das scheint der Fall zu sein, wenn der Browser nach Empfang der Date:-Headerzeile
# feststellt, daß er eine aktuelle Kopie im Chache hat. => Fehler "broken pipe"
#   => Man sollte das mit catch abfangen

# Erforderliche Pakete für wsserver
foreach {pkg_tcl pkg_debian} {html tcllib websocket tcllib httpd tcllib tls tcl-tls} {
    if {[catch {package require $pkg_tcl} errmsg]} {
        puts $errmsg
        puts "Is $pkg_debian installed?"
        exit 1
    }
}


# Der Koderumpf für den Server
namespace eval ::WSServer {

	# Default-HTTP-Serverkonfiguration
    # Das wird bei start überschrieben.
	#  port:        HTTP-Port
	#  default:     Dateiname bei Angabe eines Verzeichnisses
	#  bufsize:     Puffergröße für Requests (wichtig bei POST)
	#  maxtime:     Maximalzeit, die ein HTTP-Socket nach dem letzten Request geöffnet bleibt
    #               (10 min, Crome hat eine Verbindung nach 20 min geschlossen.)
	#  doc_root:    Stammverzeichnis für auszuliefernde Dateien
	array set HTTPD {
	    port    9900
	    default "index.html"
	    bufsize	32768
	    maxtime	600000
        doc_root "/var/www/wsserver"
	}

	# HTTP/1.[01] error codes (soweit benutzt)
	array set HTTP_ERRORS {
	    204 {No Content}
	    400 {Bad Request}
	    404 {Not Found}
	    408 {Request Timeout}
	    411 {Length Required}
	    419 {Expectation Failed}
	    500 {Internal Server Error}
	    501 {Not Implemented}
	    503 {Service Unavailable}
	    504 {Service Temporarily Unavailable}
	}

	###{{{ Hilfsfunktionen
	
	# Decode url-encoded strings.
	proc HttpdCgiMap {data} {; #{{{
	    regsub -all {([][$\\])} $data {\\\1} data
	    regsub -all {%([0-9a-fA-F][0-9a-fA-F])} $data  {[format %c 0x\1]} data
	    return [subst $data]
	    #}}}
	}; # proc HttpdCgiMap 
	
	
	# Convert a url into a pathname. (UNIX version only)
	# This is probably not right, and belongs somewhere else.
	# - Remove leading http://... if any
	# - Collapse all /./ and /../ constructs
	# - expand %xx sequences -> disallow "/"'s  and "."'s due to expansions
	proc HttpdUrl2File {root url} {; #{{{
	    regsub -all {//+} $url / url		;# collapse multiple /'s
	    while {[regsub -all {/\./} $url / url]} {}	;# collapse /./
	    while {[regsub -all {/\.\.(/|$)} $url /\x81\\1 url]} {} ;# mark /../
	    while {[regsub "/\[^/\x81]+/\x81/" $url / url]} {} ;# collapse /../
	    if {![regexp "\x81|%2\[eEfF]" $url]} {	;# invalid /../, / or . ?
		    return $root[HttpdCgiMap $url]
	    } else {
		    return ""
	    }
	    #}}}
	}; # proc HttpdUrl2File 
	
	# convert the file suffix into a mime type
	# add your own types as needed
    #TODO  application/octet-stream als default?
	array set HttpdMimeType {
	    {}		text/plain
	    .txt	text/plain
	    .html	text/html
	    .js  	text/javascript
	    .gif	image/gif
	    .jpg	image/jpeg
	    .jpeg	image/jpeg
	    .ico    image/x-icon
        .ppm    image/x-portable-pixmap
        .gz     application/gzip
        .sl3db  application/octet-stream
	}
	
	# Ermittle den Mimetyp des Dateiinhalts anhand der Endung
	# Default ist text/plain
	# @param path   Dateiname
	# @return   Http Mimetyp
	proc httpContentType {path} {; #{{{
	    variable HttpdMimeType
	
	    set type text/plain
	    catch {set type $HttpdMimeType([file extension $path])}
	    return $type
	    #}}}
	}; # proc httpContentType 
	
	
	# Datumsstring im HTTP-Format erzeugen
	# @param seconds    Entsprechende Systemuhrzeit in Sekunden
	proc httpDate {seconds} {; #{{{
	    return [clock format $seconds -format {%a, %d %b %Y %T %Z}]
	    #}}}
	}; # proc httpDate 
	
	
	# Dekodiert die Anfrageparameter und gibt sie als key/value-Paar Liste zurück.
	# (Aus tclhttpd (ein wenig modifiziert))
	# @param data   Die Anfrageparameter
    # @return       key/value-Paare in {}
    proc urlDecode {data} {; #{{{
        set data [split $data "&"]
        set result [list]
        foreach {key_value} $data {
            set l_key_value [split $key_value "="]
            #lappend result [lindex $l_key_value 0]
            set key [lindex $l_key_value 0]
            regsub -all {%([0-9a-fA-F][0-9a-fA-F])} $key  {[format %c 0x\1]} key
            set value [lindex $l_key_value 1]
            regsub -all {\+} $value "%20" value
            regsub -all {([][$\\])} $value {\\\1} value
            regsub -all {%([0-9a-fA-F][0-9a-fA-F])} $value  {[format %c 0x\1]} value
            #lappend result [encoding convertfrom utf-8 [subst $value]]
            lappend result [subst $key] [encoding convertfrom utf-8 [subst $value]]
        }
        return $result
       #}}}
    }; # proc urlDecode 

	###}}} Hilfsfunktionen
	
	
    # Die HTTP-Domänen
	namespace eval httpdomains {; #{{{
        variable httpdomains_loaded [dict create]; # Schlüssel Httpdomäne, Wert Filename
        variable wsdomains_loaded [dict create]; # Schlüssel WSdomäne, Wert Filename

        # Jede HTTP-Domäne hat ihren eigenen namespace,
        # dessen Name in der URL auf die Server-/Portspezifikation folgt.
        # Enthalten sein muß eine Prozedur "handleClientRequest", die das Handle
        # für den Clientsocket ($clientsocket), sowie die Requestparameter ($query)
        # als key/value Paare entgegennehmen kann.
        # In der 1. Zeile sollte
		# upvar #0 clientcontext$clientsocket context
        # zum Zugriff auf das Kontextarray stehen. Dieses ist initialisiert mit:
        # $context(request_headers) eben diese
        # Für die Antwort:
        # $context(contenttype) "text/html; charset=utf-8" (kann geändert werden)
        # $context(response_headers) (leeres dict für weitere Header)
        # Mit return wird die für den Client erzeugte Antwort (i.a. HTML-Text) zurückgegeben.
        # Der Aufruf erfolgt aus der Prozedur ::WSServer::httpRespond.

        # Ping-Request
        # Dient der Prüfung der Erreichbarkeit des Servers
        # Zurückgegeben wird "pong" als reiner Text
        namespace eval ping {; #{{{

            set REQUESTLOGLEVEL "Info"
	
		    # Gibt die "pong" zurück
		    proc handleClientRequest {clientsocket {query {}} } {; #{{{
		        upvar #0 clientcontext$clientsocket context
		
                set context(contenttype) "text/plain; charset=utf-8"
                #TODO Der Browser bemängelt das Fehlen der Zeichenkodierung im Header.
		        return "pong"
		        #}}}
		    }; # proc handleClientRequest
	
            #}}}
        }
	
		# Test- und Demodomäne /headers
		namespace eval headers {; #{{{

            set REQUESTLOGLEVEL "Info"
	
		    # Gibt die HTTP-Requestheader zurück
		    proc handleClientRequest {clientsocket {query {}} } {; #{{{
		        upvar #0 clientcontext$clientsocket context
		
		        set html "<!doctype html>\n"
		        append html [::html::head "Velometrik GmbH: Websocket Testseite"]
		        append html "<meta http-equiv=\"Content-Type\" content=\"text/html\" charset=\"UTF-8\"/>\n"
		        append html [::html::bodyTag]
	            append html "URL: $context(url)<BR/>"
	            append html "Query: $context(query) (Dekodiert: $query, [llength $query] Paare)<BR/>"
		        append html [::html::h3 "Header des eingegangenen HTTP-Requests"]
		        append html [::html::openTag table]
		        dict for {key value} $context(request_headers) {
		            append html [::html::row $key $value]
		        }
		        append html [::html::closeTag]; # table
		        append html [::html::end]
		        return $html
		        #}}}
		    }; # proc handleClientRequest
	
	        #}}}
	    }; # namespace eval headers 
		
		# Test- und Demodomäne /post
		namespace eval post {; #{{{

            set REQUESTLOGLEVEL "Info"
	
		    # Gibt die HTTP-Requestheader und die Postdaten zurück
		    proc handleClientRequest {clientsocket {query {}} } {; #{{{
		        upvar #0 clientcontext$clientsocket context
		
		        set html "<!doctype html>\n"
		        append html [::html::head "Velometrik GmbH: Websocket Testseite"]
		        append html "<meta http-equiv=\"Content-Type\" content=\"text/html\" charset=\"UTF-8\"/>\n"
		        append html [::html::bodyTag]
		        append html [::html::h3 "Header des eingegangenen HTTP-Requests"]
		        append html [::html::openTag table]
		        dict for {key value} $context(request_headers) {
		            append html [::html::row $key $value]
		        }
		        append html [::html::closeTag]; # table
		        append html [::html::if {$context(method) == "POST"} {
		                        postdata:<BR/>\
		                         [encoding convertto utf-8 $context(postdata)]
		                    } else {
		                        <B>Fehler</B>: POST statt $context(method) erwartet!\n
		                    }]
		        append html [::html::end]
		        return $html
		        #}}}
		    }; # proc Post
	
	        #}}}
	    }; # namespace eval post 

        #{{{ Hinzuladen der anwendungsspezifischen HTTP-Domains
        set scriptdir [file dirname [info script]]
        set httpdomainsdir [string range $scriptdir 0 [string last "/" $scriptdir]]custom/wsserver/httpdomains
        if {[file isdirectory $httpdomainsdir]} {
            foreach filename [glob -nocomplain -directory $httpdomainsdir *.tcl] {
                set httpdomain [string range [file tail $filename] 0 end-4]
	            if {[catch {source $filename} result]} {
                    srvLog {} Error "$result loading '$filename'"
                } else {
                    if {[namespace exists $httpdomain]} {
                        if {[llength [info procs ${httpdomain}::handleClientRequest]] == 0} {
                            srvLog {} Error "$filename doesn't define proc ::handleClientRequest"
                        } else {
                            if {![info exists ${httpdomain}::REQUESTLOGLEVEL]} {
                                set ${httpdomain}::REQUESTLOGLEVEL "Info"
                            }
                            dict set httpdomains_loaded $httpdomain $filename
                            srvLog {} Notice "loaded httpdomain '$httpdomain' (REQUESTLOGLEVEL [set ${httpdomain}::REQUESTLOGLEVEL])"
                        }
                    } else {
                        srvLog {} Error "$filename doesn't define namespace $httpdomain"
                    }
                }
            }
        } else {
            srvLog "[namespace current]" Warn "Directory $httpdomainsdir doesn't exist."
        }
        #}}} Hinzuladen der anwendungsspezifischen HTTP-Domains

        # Zuvor geladene Httpdomäne neu laden
        proc reload {httpdomain} {; #{{{
            variable httpdomains_loaded

            if {[dict exists $httpdomains_loaded $httpdomain]} {
                set filename [dict get $httpdomains_loaded $httpdomain]
                if {[file exists $filename]} {
                    if {[catch {source $filename} msg]} {
                        return "Error reloading ${httpdomain}:\n$msg"
                    }
                    return "$httpdomain reloaded from ${filename}."
                } else {
                    return "Cannot reload ${httpdomain}: $filename doesn't exist any more."
                }
            } else {
                return "Cannot reload ${httpdomain}: Domain wasn't loaded before."
            }
            #}}}
        }; # proc reload


	    # Gibt es die angegebene Domäne ?
	    # @param    domain  Domäne wie in der URL angegeben (/<domainname>)
	    # @param    var_procname
	    # @return   1 falls die Domäne existiert
	    #           0 sonst
	    proc exists {domain {var_procname {}} } {; #{{{
	        if {$var_procname != ""} {
	            upvar $var_procname procname
	        }
	        set domainname [string trim $domain "/"]
	        # /<unterverzeichnisse> entfernen
	        set idx [string first "/" $domainname]
	        if {$idx > 0} {
	            incr idx -1
	            set domainname [string range $domainname 0 $idx]
	        }
	        set procname "[namespace current]::${domainname}::handleClientRequest"
	        foreach child [namespace children [namespace current]] {
	            if {[namespace tail $child] == $domainname} {
	                return 1
	            }
	        }
	        return 0
	        #}}}
	    }; # proc exists
	
	    #}}}
	}; # namespace eval httpdomains 
	
	
    # Die Websocket-Domänen
	namespace eval wsdomains {; #{{{
        variable wsdomains_loaded [dict create]; # Schlüssel WSdomäne, Wert Filename

        # Testdomäne /logme
	    namespace eval logme {; #{{{
            # Variablen für diese Domäne
            # variable ...
	
	        # Mitteilung vom Client verarbeiten
            # Hinweis: Das ist ein Callback, das mit ::websocket::live übergeben wird.
            # Es wird von ::websocket aufgerufen und befindet sich deshalb in einem
            # anderen namespace Kontext.
            # Prozeduren aus ::WSServer müssen deshalb mit vollem Namespacepfad
            # aufgerufen werden.
	        proc handleClientMessage {clientsocket type msg} {; #{{{
		        upvar #0 clientcontext$clientsocket context
                # variable ...
		
		        # type laut API:
		        #   text        Complete text message
		        #   binary      Complete binary message
		        #   ping        Incoming ping message
		        #   connect     Notification of successful connection to server
		        #   disconnect  Disconnection from remote end
		        #   close       Pending closure of connection
		
		        switch -glob -nocase -- $type {
		            connect {
		                srvLog $clientsocket Info "Client connected"
                        # Kein Handlungsbedarf
		            }
		            text {
		                srvLog $clientsocket Info "text message received:\n'$msg'"
		                # => z.B. Meldung mit $msg an alle Websocket-Clients verteilen
	                    ::WSServer::disposeServerMessage logme text $msg
		            }
		            binary {
		                srvLog $clientsocket Info "binary message received"
		            }
		            ping -
		            pong {
		                srvLog $clientsocket Debug "Client $type received"
                        # Kein Handlungsbedarf
		            }
		            close {
					    srvLog $clientsocket Info "WS-Connection closing (%s)" $context(ipaddr)
                        # Kein Handlungsbedarf
		            }
		            disconnect {
					    srvLog $clientsocket Info "WS-Connection disconnected (%s)" $context(ipaddr)
                        # Das ist notwendig:
		                ::WSServer::wsClientKill $clientsocket
	                }
		            default {
					    srvLog $clientsocket Info "WS-Connection unknown message type: %s" $type
                        # Kein Handlungsbedarf
		            }
		        }
		        #}}}
	        }; # proc handleClientMessage 
	
	        #}}}
	    }; # namespace eval logme 

        #{{{ Hinzuladen der anwendungsspezifischen WS-Domains
        # Die Prozedur ermöglicht rekursives Laden aus den Unterverzeichnissen.
        proc loadWSDomains {wsdomainsdir base_domain} {
            variable wsdomains_loaded

            srvLog "[namespace current]" Debug "load WSDomains from $wsdomainsdir"
            # Zunächst die Dateien
            foreach filename [glob -nocomplain -directory $wsdomainsdir *.tcl] {
                set wsdomain ${base_domain}[string range [file tail $filename] 0 end-4]
	            if {[catch {source $filename} result]} {
                    srvLog "[namespace current]" Error "$result loading '$filename'"
                } else {
                    if {[namespace exists $wsdomain]} {
                        if {[llength [info procs ${wsdomain}::handleClientMessage]] == 0} {
                            srvLog "[namespace current]" Error "$filename doesn't define proc ::handleClientMessage"
                        } else {
                            dict set wsdomains_loaded $wsdomain $filename
                            srvLog "[namespace current]" Notice "loaded wsdomain '$wsdomain'"
                        }
                    } else {
                        srvLog "[namespace current]" Error "$filename doesn't define namespace $wsdomain"
                    }
                }
            }
            # Jetzt die Unterverzeichnisse
            foreach dirname [glob -nocomplain -type {d} -tails -directory $wsdomainsdir *] {
                if {"$base_domain" != ""} {
                    set base_domain ${base_domain}/
                }
                loadWSDomains ${wsdomainsdir}/${dirname} ${base_domain}${dirname}/
            }
        }; # proc loadWSDomains 

        set scriptdir [file dirname [info script]]
        set wsdomainsdir [string range $scriptdir 0 [string last "/" $scriptdir]]custom/wsserver/wsdomains
        if {[file isdirectory $wsdomainsdir]} {
            loadWSDomains $wsdomainsdir ""
        } else {
            srvLog "[namespace current]" Warn "Directory $wsdomainsdir doesn't exist."
        }
        #}}} Hinzuladen der anwendungsspezifischen WS-Domains

	
	    # Gibt es die angegebene Domäne ?
	    # @param    domain  Domäne wie in der URL angegeben (/<domainname>)
	    # @param    var_procname
	    # @return   1 falls die Domäne existiert
	    #           0 sonst
	    proc exists {domain {var_procname {}} } {; #{{{
	        if {$var_procname != ""} {
	            upvar $var_procname procname
	        }
	        set domainname [string trim $domain /]
	        set procname "[namespace current]::${domainname}::handleClientMessage"
	        foreach child [namespace children [namespace current]] {
	            if {[namespace tail $child] == $domainname} {
	                return 1
	            }
	        }
	        return 0
	        #}}}
	    }; # proc exists
	
	    #}}}
	}; # namespace eval wsdomains 
	
	
	# Mitteilung an die Clients der angegebenen Websocketdomäne verteilen
	# @param domain Websocket Domäne in der verteilt werden soll
    # @param type   Messagetype (text binary ping connect disconnect close)
    # @param msg    Die Mitteilung
	proc disposeServerMessage {domain type msg} {; #{{{
	    variable wsdomainclients
	
        srvLog [namespace current] Debug "disposeServerMessage $domain $type '$msg' to [array get wsdomainclients]"
	    foreach clientsocket $wsdomainclients($domain) {
		    upvar #0 clientcontext$clientsocket context
	
	        ::websocket::send $clientsocket $type $msg
	    }
        #}}}
	}; # proc disposeServerMessage 
	
	
	# Das letzte Request ist nicht innerhalb der vorgegebenen Zeit angekommen.
	# (proc HttpdTimeout  in minihttp1.1)
	# Das wird mit clientReset gesetzt, was vor dem ersten und jedem Request passiert.
	# (Nur für HTTP-Clients verwendet. Bei WS-Clients kümmert sich die Bibliothek darum.)
	proc clientTimeout {clientsocket} {; #{{{
	    upvar #0 clientcontext$clientsocket context
	
	    if {$context(state) == "start" && $context(requestcount)>0} {
	        # Es wurde bereits ein Request ausgeführt und auf ein weiteres gewartet.
	        # Das ist kein Fehler.
	        srvLog $clientsocket Notice "Timeout: HTTP-Socket for $context(ipaddr):$context(port) closed"
	    } else {
	        # Irgendetwas ist nicht wie gedacht gelaufen.
	        srvLog $clientsocket Error "Timeout bevor first request from $context(ipaddr)"
	    }
	    close $clientsocket
	    unset context
	    #}}}
	}; # proc clientTimeout 
	
	
	# HTTP-Client komplett entfernen:
	#  Timeout löschen, sofern gesetzt
	#  Socket schließen
	#  Kontext löschen
	proc httpClientKill {clientsocket} {; #{{{
	    upvar #0 clientcontext$clientsocket context
	
	    # clientTimout ist obsolet.
	    if {[info exists context(cancel)]} {
	        # Wenn $context(cancel) ungültig ist, tut after cancel nichts.
	        after cancel $context(cancel)
	    }
	    if {[catch {close $clientsocket} result]} {
	        srvLog $clientsocket Error "HTTP-Socket close error: $result"
	    } else {
	        srvLog $clientsocket Info "HTTP-Socket closed."
	    }
	    unset context
	    #}}}
	}; # proc httpClientKill
	
	
	# WS-Client komplett entfernen:
	#  Socket schließen
	#  Aus wsdomainclients entfernen
	#  Kontext löschen
	proc wsClientKill {clientsocket} {; #{{{
	    upvar #0 clientcontext$clientsocket context
	    variable wsdomainclients
	
	    if {[catch {close $clientsocket} result]} {
	        srvLog $clientsocket Error "WS-Socket close error: $result"
	    } else {
	        srvLog $clientsocket Notice "WS-Socket closed."
	    }
	    # Socket aus den Websocketclients entfernen
	    foreach wsdomain [array names wsdomainclients] {
	        set idx [lsearch $wsdomainclients($wsdomain) $clientsocket]
	        if {$idx >= 0} {
	            set wsdomainclients($wsdomain) [lreplace $wsdomainclients($wsdomain) $idx $idx]
	            srvLog $clientsocket Debug "Socket aus wsdomainclients($wsdomain) entfernt."
	            
	        }
	    }
	    unset context
	    #}}}
	}; # proc wsClientKill
	
	
	# Requeststatus für einen Client zurücksetzen
	# (proc HttpdReset in "minihttpd1.1")
	proc clientReset {clientsocket} {; #{{{
	    upvar #0 clientcontext$clientsocket context
	    variable HTTPD
	
	    # Die Zeit bis zum Timeot des Requests läuft wieder von vorn.
	    if {[info exists context(cancel)]} {
	        after cancel $context(cancel)
	    }
	    foreach key {method url} {
	        set context($key) ""
	    }
	    set context(linemode) 1 
	    set context(request_headers) [dict create]
	    set context(state) "start"
	    set context(cancel) [after $HTTPD(maxtime) [list [namespace current]::clientTimeout $clientsocket]]
	    fconfigure $clientsocket -translation {auto crlf}
	    #}}}
	}; # prod clientReset
	
	
	# Mit Fehlermeldung antworten
	# (Schließt und löscht den Socket)
    # Am Ende kehrt die Prozedur zurück zum Aufrufer.
    # TODO Mit return -level 2 wäre die Rückkehr vermeidbar, was den Kode im Aufrufer vereinfacht.
	proc httpRespondError {clientsocket httpcode args} {; #{{{
	    upvar #0 clientcontext$clientsocket context
	    variable HTTP_ERRORS
	
	    set HttpdErrorFormat {
	        <title>Error: %1$s</title>
	        Got the error: <b>%2$s</b><br>
	        while trying to obtain <b>%3$s</b>
	    }
	    set message [format $HttpdErrorFormat $httpcode $HTTP_ERRORS($httpcode) $context(url)]
	    append head "HTTP/1.$context(httpversion) $httpcode $HTTP_ERRORS($httpcode)\n"  
	    append head "Date: [httpDate [clock seconds]]\n"
	    append head "Connection: close"  \n
	    append head "Content-Length: [string length $message]\n"
	
	    # Because there is an error condition, the socket may be "dead"
	    catch {
		    fconfigure $clientsocket -translation crlf
		    puts -nonewline $clientsocket $head\n$message
		    flush $clientsocket
	    } reason
        if {[info level] > 1} {
            # Hinweis auf die aufrufende Prozedur
            set from "from [info level -1] "
        } else {
            set from ""
        }
	    srvLog $clientsocket Error "${from}$httpcode $HTTP_ERRORS($httpcode) $args $reason"
	    # Socket serverseitig schließen und entfernen
	    httpClientKill $clientsocket
	    #}}}
	}; # proc httpRespondError 
	

    # Domain-Request abschließen
    # clientsocket  Sockethandler des Clients
    # html          Vom Domainhandler gelieferte Daten
    proc finishDomainResponse {clientsocket html} {; #{{{
	    upvar #0 clientcontext$clientsocket context

		puts $clientsocket "HTTP/1.$context(httpversion) $context(responsecode)"
		puts $clientsocket "Date: [httpDate [clock seconds]]"
		puts $clientsocket "Last-Modified: [httpDate [clock seconds]]"
		puts $clientsocket "Content-Type: $context(contenttype)"
		puts $clientsocket "Content-Length: [string length $html]"
        # Zusätzliche Header (vom Domainhandler ergänzt)
        dict for {key value} $context(response_headers) {
		    puts $clientsocket "$key: $value"
        }
		puts $clientsocket ""
		flush $clientsocket
		
		if {$context(method) != "HEAD"} {; # Daten weitergeben.
            #TODO -encoding hängt von der Domäne ab und sollte dort definiert werden.
            # (UTF-8 gilt nur bei Text.)
            # fconfigure $clientsocket -translation binary -encoding utf-8
            fconfigure $clientsocket -translation binary
            puts -nonewline $clientsocket $html
            flush $clientsocket
	    }
	    clientReset $clientsocket

        #}}}
    }; # proc finishDomainResponse 


    # Domain-Request ausführen
    # clientsocket  Sockethandler des Clients
    # domainhandler Erzeuger der Rückgabedaten
    proc httpRespondDomain {clientsocket domainhandler} {; #{{{
	    upvar #0 clientcontext$clientsocket context

        # Das kann der Domainhandler ersetzen:
	    set context(contenttype) "text/html; charset=utf-8"
			if {$context(method) == "HEAD"} {
	        set context(responsecode) "204 No Content"
	    } else {
	        set context(responsecode) "200 Data follows"
	    }
        # Zusätzliche Header:
	    set context(response_headers) [dict create]

        # Domainhandler aufrufen
        srvLog $clientsocket Debug "httpRespond mit $domainhandler"
        set context(self_finish) 0
	    set html [$domainhandler $clientsocket [urlDecode $context(query)]]
	
        if {![info exists context(httpversion)]} {
            #TODO Wie kann das passieren ?
            srvLog $clientsocket Info "httpRespond: context gelöscht => Abbruch"
            return
        }

        if {$context(self_finish)} {; # Domäne beendet den Request selbst.
            srvLog $clientsocket Info "$domainhandler will finish request."
            return
        }

        finishDomainResponse $clientsocket $html
        # Fertigmeldung wie für den Handler gesetzt:
		srvLog $clientsocket [set [regsub {^(.*)handleClientRequest$} $domainhandler {\1::REQUESTLOGLEVEL}]] "Done %d Bytes" [string length $html]
	
        #}}}
    }; # proc httpRespondDomain 
	
	# Interpretiere die URL als TCL-String
    # Holt den Dateiinhalt, schickt ihn mit dem subst-Befehl
    # durch den TCL-Interpreter und gibt das Ergebnis zurück.
    # Weitere Details s. htdocs/doc/demo_tcls.tcls
    # @param clientsocket  Sockethandler des Clients
    proc httpRespondTCLS {clientsocket} {; #{{{
	    upvar #0 clientcontext$clientsocket context
	    variable HTTPD
	
	    set localpath [HttpdUrl2File $HTTPD(doc_root) $context(url)]
	    if {![file readable $localpath]} {
	        httpRespondError $clientsocket 404 $context(url)
        } else {; # Datei interpretieren und Ergebnis ausliefern
            #TODO Dateityp analysieren und in Header einbauen (html, js, ...)
            # Datei holen
            if {[catch {
                    set fd_tcls [open $localpath r]
                    set tcls [read $fd_tcls]
                    close $fd_tcls
                } error]} {
	            httpRespondError $clientsocket 404 $::errorInfo
                return
            }
            # Datei interpretieren
            catch {namespace delete ::tclsrequest}
            namespace eval ::tclsrequest {; # Request-namespace anlegen
                #TODO Damit ist WSServer nicht mehr anwendungsneutral:
                # Import von Kommandos aus dem DB-Kernelmodul
                if {[namespace exists ::kernel::DB]} {
                    namespace import ::kernel::DB::currentTimestamp
                    namespace import ::kernel::DB::currentDate
                    namespace import ::kernel::DB::nextId
                }
            }
            set ::tclsrequest::_tcls $tcls
            set ::tclsrequest::localdir [file dirname $localpath]
            #TODO Damit ist WSServer nicht mehr anwendungsneutral:
            # Import von Kommandos aus dem DB-Kernelmodul
            if {[namespace exists ::kernel::DB]} {
                set ::tclsrequest::dbClients $::kernel::DB::clients
            }
            set ::tclsrequest::query [urlDecode $context(query)]
            set ::tclsrequest::texttype "html"
            set ::tclsrequest::clientsocket $clientsocket; # nicht dokumentiert
            namespace eval ::tclsrequest {; # include - Befehl

                # Datei einfügen
                # .tcl Dateien werden mit dem source Kommando ausgewertet
                # .tcls Dateien werden mit subst ausgewertet
                # Der Inhalt anderer Dateien wird wie gelesen übergeben
                # @param filename   Einzufügende Datei
                #                   Eine absolute Pfadangabe bezieht sich auf doc_root.
                #                   Relative Pfade auf ein Parentverzeichnis sind nicht erlaubt.
                # @return   Einzufügender Inhalt
                proc include {filename} {; #{{{
                    variable clientsocket
                    variable localdir

                    set content ""
                    if {[file pathtype $filename] == "absolute"} {
                        set filename $::WSServer::HTTPD(doc_root)$filename
                    } else {; # relativ
                        if {[string range $filename 0 1] == ".."} {
                            error "$filename: subdir not allowed"
                        }
                        set filename ${localdir}/$filename
                    }
                    if {![file exists $filename]} {
	                    error "$filename not found"
                    }
                    switch [string tolower [file extension $filename]] {
                        .tcl {
                            set content [source $filename]
                        }
                        .tcls {
                            set fd [open $filename r]
                            set content [subst [read $fd]]
                            catch {close $fd}
                        }
                        default {
                            set fd [open $filename r]
                            set content [read $fd]
                            catch {close $fd}
                        }
                    }
                    return $content
                    #}}}
                }; # proc include

            }
            if {[catch {
				    namespace eval ::tclsrequest {
				        set html [subst $_tcls]
                    }
                } error]} {
                # => Internal Server Error
	            httpRespondError $clientsocket 500 $::errorInfo
                return
            }
            # Ergebnis ausliefern
			if {$context(method) == "HEAD"} {
	            set responsecode "204 No Content"
	        } else {
	            set responsecode "200 Data follows"
	        }
	        set content_length0 [string length $::tclsrequest::html]
	        set content_length [string length [encoding convertto utf-8 $::tclsrequest::html]]
			puts $clientsocket "HTTP/1.$context(httpversion) $responsecode"
			puts $clientsocket "Date: [httpDate [clock seconds]]"
			puts $clientsocket "Last-Modified: [httpDate [file mtime $localpath]]"
			puts $clientsocket "Content-Type: text/${::tclsrequest::texttype}; charset=utf-8"
			puts $clientsocket "Content-Length: $content_length"
		
			puts $clientsocket ""
			flush $clientsocket
		
			if {$context(method) != "HEAD"} {; # Dateiinhalt weitergeben.
				fconfigure $clientsocket -translation binary -encoding utf-8
			    puts -nonewline $clientsocket $::tclsrequest::html
			}
	        srvLog $clientsocket Info "Done %d (%d) Bytes with .tcls" $content_length $content_length0
			clientReset $clientsocket
        }; # else interpretieren und ausgeben
        #}}}
    }; # proc httpRespondTCLS 


    # File-Request abschließen
    # @param clientsocket   Sockethandler des Clients
    # @param fd             Lesedescriptor der Datei
    # @param content_length Länge lt. HTTP-Header
    # @param n_bytes        Tatsächliche Länge
    # @param errmsg         Evtl. Fehlermeldung
    proc finishFileResponse {clientsocket fd content_length n_bytes {errmsg {}} } {; #{{{
        catch {close $fd}
        if {"$errmsg" == ""} {
	        srvLog $clientsocket Info [format "Done %d(%d)  Bytes" $n_bytes $content_length]
        } else {
	        srvLog $clientsocket Error [format "%s\n%d(%d) Bytes transferred" $errmsg $n_bytes $content_length]
        }
        clientReset $clientsocket
        #}}}
    }; # proc finishFileResponse 


	# Interpretiere die URL als Dateiname
	proc httpRespondFile {clientsocket} {; #{{{
	    upvar #0 clientcontext$clientsocket context
	    variable HTTPD
	
	    set localpath [HttpdUrl2File $HTTPD(doc_root) $context(url)]
	    if {[string length $localpath] == 0} {
	        httpRespondError $clientsocket 400
	    } elseif {![file readable $localpath]} {
	        httpRespondError $clientsocket 404
	    } else {; # Datei ausliefern
			if {$context(method) == "HEAD"} {
	            set responsecode "204 No Content"
	        } else {
	            set responsecode "200 Data follows"
	        }
	        set content_length [file size $localpath]
			puts $clientsocket "HTTP/1.$context(httpversion) $responsecode"
			puts $clientsocket "Date: [clock format [file mtime $localpath] -format {%a, %d %b %Y %T %Z}]"
			puts $clientsocket "Last-Modified: [httpDate [file mtime $localpath]]"
			puts $clientsocket "Content-Type: [httpContentType $localpath]"
			puts $clientsocket "Content-Length: $content_length"
		
			puts $clientsocket ""
			flush $clientsocket
		
			if {$context(method) != "HEAD"} {; # Dateiinhalt weitergeben.
			    set in [open $localpath]
			    fconfigure $clientsocket -translation binary
			    fconfigure $in -translation binary
			    fcopy $in $clientsocket -command "[namespace current]::finishFileResponse $clientsocket $in $content_length"
                return ""
			}
	        srvLog $clientsocket Info [format "Done %d Bytes" $content_length]
			clientReset $clientsocket
            return ""
		
	    }; # else Datei ausliefern
	    #}}}
	}; # proc httpRespondFile 
	
	
	# Die Antwort auf ein HTTP-Request ausliefern
    # Bestimmt die Art der Requestantwort und startet deren Erstellung.
	proc httpRespond {clientsocket} {; #{{{
	    upvar #0 clientcontext$clientsocket context
	    variable HTTPD

	    if {[httpdomains::exists $context(url) domainhandler]} {
            # Request wird vom Domainhandler ausgeführt.
            httpRespondDomain $clientsocket $domainhandler
        } else {
	        if {[file isdirectory $context(url)]} {
                if {[string index $context(url) end] == "/"} {
	                append context(url) $HTTPD(default)
                } else {
	                append context(url) / $HTTPD(default)
                }
                srvLog $clientsocket Notice "Try default: $context(url)"
            }
            if {"[file extension $context(url)]" == ".tcls"} {; # TCL-String-Datei
                httpRespondTCLS $clientsocket
    	    } else {; # weder Domäne noch TCL-String Datei
    	        # => Filezugriff probieren
    	        httpRespondFile $clientsocket
    	    }
	    }
	    #}}}
	}; # proc httpRespond 
	
	
	# Vom einem Client, dessen Verbindungsanforderung mit serverConnect
	#   bearbeitet wurde, sind Daten angekommen.
    # Liest die nächste Headerzeile und startet httpRespond,
    # sobald dieser vollständig ist.
	proc httpAccept {clientsocket} {; #{{{
	    upvar #0 clientcontext$clientsocket context
        variable serversocket 
	    variable wsdomainclients
	
	    if {$context(linemode)} {
	        # Im Fehlerfall wird not_blank -1
	        set not_blank [string compare [gets $clientsocket line] 0]
	        if {$not_blank == -1} {; #{{{ nichts gelesen
	            if {[eof $clientsocket]} {
	                # Socket wurde clientseitig geschlossen
	                if {$context(state) == "start" && $context(requestcount)>0} {
	                    # Mindestens ein Request wurde für $sock bereits bearbeitet.
				        srvLog $clientsocket Notice "HTTP Connection closed by client (%s)" $context(ipaddr)
	                } else {
	                    # Request noch nicht oder zuletzt unvollständig bearbeitet.
				        srvLog $clientsocket Error "Broken HTTP connection fetching request (state=$context(state), requestcount=$context(requestcount))"
	                }
	                # Socket serverseitig schließen und entfernen
	                httpClientKill $clientsocket
	            } else {
			    	srvLog $clientsocket Warning "Partial read, retrying"
	            }
	            #}}}
	        } elseif {"$context(state)" == "start"} {; #{{{ Requeststart erwartet
	            if {$not_blank} {; # Requestzeile
					if {[regexp {(HEAD|POST|GET) ([^?]+)\??([^ ]*) HTTP/1.([01])} $line \
						dummy context(method) context(url) context(query) context(httpversion)]} {
                        if {[regexp {rnd=[0-9]+} $line]} {
	                        srvLog $clientsocket Debug $line
                        } else {
	                        srvLog $clientsocket Info $line
                        }
	                    set context(state) "headers"
	                } else {
					    httpRespondError $clientsocket 400 $line
	                }
	            } else {; # Leere erste Zeile
				    srvLog $clientsocket Warning "Initial blank line fetching HTTP request"
	            }
	            #}}}
	        } else {; #{{{ (Weitere) Headerzeile oder -ende
	            if {$not_blank} {; #{{{ (Weitere) Headerzeile
					if {[regexp {([^:]+):[ 	]*(.*)}  $line {} key value]} {
					    #??? set key [string tolower $key]
					    if {[dict exists $context(request_headers) $key]} {
						    dict append context(request_headers) $key ", $value"
					    } else {
	                        dict set context(request_headers) $key $value
					    }
					} else {
					    httpRespondError $clientsocket 400 $line
					}
	                #}}}
	            } else {; #{{{ Headerende
	                # Handelt es sich um ein Websocket-Request ?
	                if {[wsdomains::exists $context(url) wshandler]} {; #{{{ ws-Domäne
	                    # Zugriff auf eine Websocket-Domäne => Gültigkeit prüfen
                        if {[catch {::websocket::test $serversocket $clientsocket $context(url) $context(request_headers)} testresult]} {
                            # Fehler, wenn kein Protokoll angegeben wurde.
		                    httpRespondError $clientsocket 404 "did not get a valid websocket request: $testresult" 
                        } else {; # ::websocket::test hat Ergebnis geliefert
		                    if {$testresult} {
		                        # Gültiges Websocket-Request
		                        ::websocket::upgrade $clientsocket; # Ruft ::websocket::takeover auf
		                        srvLog $clientsocket Notice "Websocket [::websocket::conninfo $clientsocket type] from [::websocket::conninfo $clientsocket sockname] to [::websocket::conninfo $clientsocket peername] established."
		                        fileevent $clientsocket readable; # HTTP-Handler deaktivieren
		                        # Kein clientTimout mehr.
		                        if {[info exists context(cancel)]} {
		                            after cancel $context(cancel)
		                        }
		                        # Registrieren den Client als zu der von der angegebenen url gehörigen Domäne gehörig.
		                        lappend wsdomainclients([string trim $context(url) /]) $clientsocket
		                    } else {; # Kein gültiges Websocket-Request
		                        # Fehlermeldung und Socket schließen
		                        set headers $context(request_headers); # context ist nach httpRespondError nicht mehr vorhanden.
		                        httpRespondError $clientsocket 404 "did not get a valid websocket request: url : $context(url)" 
		                        set errormessage "Header of Invalid websocket request:"
		                        dict for {key value} $headers {
		                            append errormessage "\n$key $value"
		                        }
		                        srvLog $clientsocket Debug $errormessage
		                    }
                        }; # else ::websocket::test hat Ergebnis geliefert
                        #}}}
	                } else {; #{{{ Httprequest
			            if {$context(method) == "POST" && [dict exists $context(request_headers) "Content-Length"]} {; #{{{
	                        # POST-Request mit (notwendiger) Längenangabe
	                        # => Umschalten auf "Postdaten lesen"
				            set context(linemode) 0; # POST-Daten als Block lesen
			                set context(count) [dict get $context(request_headers) "Content-Length"]
	                        srvLog $clientsocket Info "$context(count) bytes of POST-data expected"
	                        set context(postdata) ""
			                if {$context(httpversion) && [dict exists $context(request_headers) "Expect"]} {
	                            # Der HTTP 1.1 Client erwartet ...
			                    if {[dict get $context(request_headers) "Expect"] == "100-continue"} {
	                                # ... ein Startsignal für die Daten.
					                puts $clientsocket "100 Continue HTTP/1.1\n"
					                flush $clientsocket
					            } else {
	                                # ... etwas, das wir nicht kennen.
					                httpRespondError $clientsocket 419 [dict get $context(request_headers) "Expect"]
					            }
	                        }
				            fconfigure $clientsocket -translation {binary crlf}
                            #}}}
	                    } elseif {$context(method) != "POST"} {
                            #TODO catch (Hier "broken pipe" abfangen. s. oben)
                            # Msg. z.B.: 'error writing "sock558b4b2b10": broken pipe'
	                        httpRespond $clientsocket; # GET oder HEAD
			            } else {
	                        # POST ohne content-length
				            httpRespondError $clientsocket 411 "Confusing mime headers"
			            }
	                    #}}}
	                }; # else Headerende
	                incr context(requestcount)
	                #}}}
	            }; # else Headerende
	            #}}}
	        }; # else (Weitere) Headerzeile oder -ende
	    } elseif {![eof $clientsocket]} {; #{{{ (Weitere) POST-Daten entgegennehmen
	        srvLog $clientsocket Info "POST-data received"
	        append context(postdata) [read $clientsocket $context(count)]
	        set context(count) [expr {[dict get $context(request_headers) "Content-Length"] - [string length $context(postdata)]}]
	        srvLog $clientsocket Info "$context(count) bytes of POST-data remaining"
	        if {$context(count) <= 0} {
		        httpRespond $clientsocket; # POST
		    }
	        #}}}
	    } else {; #{{{ Verbindung beim Lesen der POST-Daten unterbrochen
		    srvLog $clientsocket Error "Broken connection reading POST data"
	        httpClientKill $clientsocket
	        #}}}
	    }; # if linemode ...
	    #}}}
	}; # proc httpAccept 
	
	
	# Eine Verbindunganforderung an den Server bearbeiten
    # Konfiguriert clientsocket, erstellt den Clientcontext
    # und aktiviert httpAccept für die Entgegennahme des erwarteten HTTP-Requests.
	proc serverConnect {clientsocket clientip clientport} {; #{{{
	    upvar #0 clientcontext$clientsocket context
	    variable HTTPD
	
	    srvLog $clientsocket Notice "Connect from ${clientip}:${clientport}"
	    # Den Socket für die Datenübernahme konfigurieren
	    fconfigure $clientsocket -blocking 0 -buffersize $HTTPD(bufsize) -buffering line
	    # Kontext für diesen Client einrichten
	    array set context [list httpversion 0 requestcount 0 ipaddr $clientip port $clientport]
	    clientReset $clientsocket
	    # httpAccept aufrufen, sobald Daten anliegen
	    fileevent $clientsocket readable [list [namespace current]::httpAccept $clientsocket]
	    #}}}
	}; # proc serverConnect 


    # Den Server starten
    # @param httpd_config   Einzustellende Konfigurationsparameter als key/value Liste
    #                       Die möglichen Parameternamen und ihre Bedeutung finden sich
    #                       oben bei HTTPD.
    proc start {httpd_config} {; #{{{
        variable serversocket
        variable HTTPD

        array set HTTPD $httpd_config
        if {![file isdirectory $HTTPD(doc_root)]} {
            error "doc_root '$HTTPD(doc_root)' is not a directory"
        }
	    # Serversocket erstellen
	    set serversocket [socket -server [namespace current]::serverConnect $HTTPD(port)]
	    # Den Serversocket zum Websocket Server machen
	    ::websocket::server $serversocket
	    # Dem Websocket Server alle Domains mitteilen.
		foreach wsdomain [namespace children "wsdomains"] {
	        set wsdomain [namespace tail $wsdomain]
	        ::websocket::live $serversocket /$wsdomain [namespace current]::wsdomains::${wsdomain}::handleClientMessage
	    }
        srvLog [namespace current]::start Info "Options: [array get HTTPD]"
        #}}}
    }; # proc start
	
	# Die Clients der einzelnen Websocketdomains
	# Das ist ein Array mit dem Domainnamen als Index und
	# Listen der zugehörigen Clientsocketnamen als Wert.
	# Hier erfolgt die Initialisierung mit leeren Listen.
	foreach wsdomain [namespace children "wsdomains"] {
        set wsdomain [namespace tail $wsdomain]
	    set wsdomainclients($wsdomain) [list]
        srvLog [namespace current] Notice "WS domain /$wsdomain established."
	}

}; # namespace eval ::WSServer

